import { ApplicationWrapper, MediaAttributes } from './interfaces';
import { RendererFactory } from '../renderers/interfaces';
import { DOMRendererFactory } from '../renderers/dom-renderer-factory';

/**
 * Plain implementation of ApplicationWrapper that uses native browser APIs.
 * 
 * This is not used by Drupal, but can be used by other applications to integrate ProseMirror.
 * Will be moved to a separate node module in the future.
 */
export abstract class PlainWrapper implements ApplicationWrapper {
  private rendererFactory: RendererFactory = new DOMRendererFactory();
  dialogSettings = {
    dialogClass: '',
    autoResize: window.matchMedia('(min-width: 600px)').matches,
    width: 'auto'
  };

  /**
   * Select media from the media library
   */
  abstract selectMedia(callback: (attributes: MediaAttributes) => void): void;

  /**
   * Get media preview
   */
  abstract getMediaPreview(uuid: string): Promise<{ preview: string, label: string | null }>;

  openModalDialog(url: string, saveCallback: (data: any) => void, dialogSettings: Record<string, any>) {
    const iframe = document.createElement('iframe');
    iframe.src = this.sanitizeUrl(url);
    iframe.className = 'prosemirror-iframe';

    const dialog = document.createElement('div');
    dialog.className = 'prosemirror-dialog';
    if (dialogSettings.width) dialog.style.width = dialogSettings.width;
    if (dialogSettings.height) dialog.style.height = dialogSettings.height;

    const header = document.createElement('div');
    header.className = 'prosemirror-dialog-header';

    const title = document.createElement('h2');
    title.textContent = dialogSettings.title || 'Dialog';
    title.className = 'prosemirror-dialog-title';

    const closeButton = document.createElement('button');
    closeButton.textContent = '×';
    closeButton.className = 'prosemirror-dialog-close';

    const content = document.createElement('div');
    content.className = 'prosemirror-dialog-content';
    content.appendChild(iframe);

    header.appendChild(title);
    header.appendChild(closeButton);
    dialog.appendChild(header);
    dialog.appendChild(content);

    const overlay = document.createElement('div');
    overlay.className = 'prosemirror-dialog-overlay';

    document.body.appendChild(overlay);
    document.body.appendChild(dialog);

    const close = () => {
      document.body.removeChild(overlay);
      document.body.removeChild(dialog);
      if (saveCallback) {
        saveCallback({});
      }
    };

    closeButton.onclick = close;
    overlay.onclick = close;

    // Listen for messages from the iframe
    window.addEventListener('message', (event) => {
      if (event.data && event.data.dialogClose) {
        close();
        if (saveCallback) {
          saveCallback(event.data);
        }
      }
    });
  }

  openCustomDialog(content: HTMLElement, saveCallback: (data: any) => void, dialogSettings: Record<string, any>) {
    const dialog = document.createElement('div');
    dialog.className = 'prosemirror-dialog';
    if (dialogSettings.width) dialog.style.width = dialogSettings.width;
    if (dialogSettings.height) dialog.style.height = dialogSettings.height;

    const header = document.createElement('div');
    header.className = 'prosemirror-dialog-header';

    const title = document.createElement('h2');
    title.textContent = dialogSettings.title || 'Dialog';
    title.className = 'prosemirror-dialog-title';

    const closeButton = document.createElement('button');
    closeButton.textContent = '×';
    closeButton.className = 'prosemirror-dialog-close';

    const dialogContent = document.createElement('div');
    dialogContent.className = 'prosemirror-dialog-content';
    dialogContent.appendChild(content);

    header.appendChild(title);
    header.appendChild(closeButton);
    dialog.appendChild(header);
    dialog.appendChild(dialogContent);

    const overlay = document.createElement('div');
    overlay.className = 'prosemirror-dialog-overlay';

    document.body.appendChild(overlay);
    document.body.appendChild(dialog);

    const close = () => {
      document.body.removeChild(overlay);
      document.body.removeChild(dialog);
      if (saveCallback) {
        saveCallback({});
      }
    };

    closeButton.onclick = close;
    overlay.onclick = close;

    return {
      close,
    };
  }

  ajax(options: {
    url: string;
    type?: string;
    data?: any;
    dataType?: string;
    success?: (response: any) => void;
    error?: (error: any) => void;
    [key: string]: any;
  }): void {
    fetch(options.url, {
      method: options.type || 'GET',
      body: options.data ? JSON.stringify(options.data) : undefined,
      headers: {
        'Content-Type': 'application/json'
      }
    })
      .then(response => options.dataType === 'json' ? response.json() : response.text())
      .then(data => options.success?.(data))
      .catch(error => options.error?.(error));
  }

  addWindowEventListener(event: string, callback: (event: Event, ...args: any[]) => void): void {
    window.addEventListener(event, callback as EventListener);
  }

  removeWindowEventListener(event: string, callback: (event: Event, ...args: any[]) => void): void {
    window.removeEventListener(event, callback as EventListener);
  }

  addDocumentEventListener(event: string, callback: (event: Event, ...args: any[]) => void): void {
    document.addEventListener(event, callback as EventListener);
  }

  removeDocumentEventListener(event: string, callback: (event: Event, ...args: any[]) => void): void {
    document.removeEventListener(event, callback as EventListener);
  }

  parseHTML(html: string): HTMLElement {
    const parser = new DOMParser();
    return parser.parseFromString(html, 'text/html').body;
  }

  getElementValue(element: HTMLElement): string {
    return (element as HTMLInputElement).value;
  }

  setElementValue(element: HTMLElement, value: string): void {
    (element as HTMLInputElement).value = value;
  }

  getRendererFactory(): RendererFactory {
    return this.rendererFactory;
  }

  private sanitizeUrl(url: string): string {
    // Prevent javascript: and data: URLs which could be used for XSS
    const dangerousProtocols = ['javascript:', 'data:', 'vbscript:', 'file:', 'about:'];
    const lowerUrl = url.toLowerCase().trim();
    
    for (const protocol of dangerousProtocols) {
      if (lowerUrl.startsWith(protocol)) {
        return 'about:blank'; // Return safe fallback for iframe
      }
    }
    
    return url;
  }
}
