<?php

namespace Drupal\prosemirror_api_field\Field;

use Drupal\Core\Field\FieldItemList;
use Drupal\Core\TypedData\ComputedItemListTrait;

/**
 * Computed field item list for ProseMirror enhanced data.
 */
class ProseMirrorComputedFieldItemList extends FieldItemList {

  use ComputedItemListTrait;

  /**
   * {@inheritdoc}
   */
  protected function computeValue() {
    // Get the source field name from settings.
    $source_field = $this->getFieldDefinition()->getSetting('source_field');

    if (!$source_field) {
      return;
    }

    // Get the entity and the source field.
    $entity = $this->getEntity();

    if (!$entity || !$entity->hasField($source_field)) {
      return;
    }

    $source_field_items = $entity->get($source_field);

    foreach ($source_field_items as $delta => $field_item) {
      $values = $field_item->getValue();
      $format_id = $values['format'] ?? NULL;
      $json_content = $values['value'] ?? '';

      if ($format_id && _prosemirror_is_prosemirror_format($format_id) && !empty($json_content)) {
        $content_array = json_decode($json_content, TRUE);

        if (json_last_error() === JSON_ERROR_NONE && is_array($content_array)) {
          // Get services.
          $transformation_helper = \Drupal::service('prosemirror.transformation_helper');
          $renderer = \Drupal::service('prosemirror.renderer');
          $entity_repository = \Drupal::service('entity.repository');

          // Process the content.
          $element_instance = $transformation_helper->validateAndSanitize($content_array);
          $rendered_html = $renderer->render($content_array, 'html');

          // Normalize entity references.
          $includes = [];
          foreach ($element_instance->getReferences() as $reference) {
            try {
              $referenced_entity = $entity_repository->loadEntityByUuid(
                $reference->getEntityType(),
                $reference->getEntityUuid()
              );

              if ($referenced_entity) {
                $entity_type = $reference->getEntityType();
                $bundle = $referenced_entity->bundle();

                $includes[] = [
                  'type' => $entity_type . '--' . $bundle,
                  'id' => $reference->getEntityUuid(),
                  'meta' => [
                    'drupal_internal__target_id' => (int) $referenced_entity->id(),
                    'label' => method_exists($referenced_entity, 'label') ? $referenced_entity->label() : '',
                  ],
                ];
              }
            }
            catch (\Exception $e) {
              // Skip invalid references.
            }
          }

          // Create the computed field data.
          $computed_data = [
            'processed' => $rendered_html,
            'value' => $content_array,
            'included' => $includes,
          ];

          $this->list[$delta] = $this->createItem($delta, $computed_data);
        }
      }
    }
  }

}
