<?php

declare(strict_types=1);

namespace Drupal\prosemirror\Element;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\prosemirror\ProseMirrorElementInterface;
use Drupal\prosemirror\ProseMirrorMarkInterface;

/**
 * Provides both configurable and system ProseMirror elements and marks through a unified interface.
 */
class ElementProvider {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Cache of all elements (configurable + system).
   *
   * @var \Drupal\prosemirror\ProseMirrorElementInterface[]|null
   */
  protected ?array $elements = NULL;

  /**
   * Cache of all marks.
   *
   * @var \Drupal\prosemirror\ProseMirrorMarkInterface[]|null
   */
  protected ?array $marks = NULL;

  /**
   * Constructs an ElementProvider.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Gets an element by machine name.
   *
   * @param string $machineName
   *   The machine name.
   *
   * @return \Drupal\prosemirror\ProseMirrorElementInterface|null
   *   The element or NULL if not found.
   */
  public function getElement(string $machineName): ?ProseMirrorElementInterface {
    $elements = $this->getAllElements();
    return $elements[$machineName] ?? NULL;
  }

  /**
   * Gets a mark by machine name.
   *
   * @param string $machineName
   *   The machine name.
   *
   * @return \Drupal\prosemirror\ProseMirrorMarkInterface|null
   *   The mark or NULL if not found.
   */
  public function getMark(string $machineName): ?ProseMirrorMarkInterface {
    $marks = $this->getAllMarks();
    return $marks[$machineName] ?? NULL;
  }

  /**
   * Gets all elements (configurable + system).
   *
   * @return \Drupal\prosemirror\ProseMirrorElementInterface[]
   *   Array of elements keyed by machine name.
   */
  public function getAllElements(): array {
    if ($this->elements !== NULL) {
      return $this->elements;
    }

    $this->elements = [];

    // Load configurable elements first.
    try {
      $element_storage = $this->entityTypeManager->getStorage('prosemirror_element');
      $configurable = $element_storage->loadMultiple();

      foreach ($configurable as $element) {
        $this->elements[$element->id()] = $element;
      }
    }
    catch (\Exception $e) {
      // Log error but continue with system elements.
      \Drupal::logger('prosemirror')->error('Failed to load configurable elements: @error', [
        '@error' => $e->getMessage(),
      ]);
    }

    // Add system elements (these override configurable if same machine name)
    $system_definitions = SystemElementTypes::getDefinitions();
    foreach ($system_definitions as $machine_name => $definition) {
      // Skip if a configurable element with same name exists and is not being overridden.
      if (isset($this->elements[$machine_name]) && empty($definition['override'])) {
        continue;
      }

      $this->elements[$machine_name] = new SystemElement([
        'id' => $machine_name,
        'definition' => $definition,
      ], 'prosemirror_element');
    }

    return $this->elements;
  }

  /**
   * Gets all marks.
   *
   * @return \Drupal\prosemirror\ProseMirrorMarkInterface[]
   *   Array of marks keyed by machine name.
   */
  public function getAllMarks(): array {
    if ($this->marks !== NULL) {
      return $this->marks;
    }

    $this->marks = [];

    // Load configurable marks.
    try {
      $mark_storage = $this->entityTypeManager->getStorage('prosemirror_mark');
      $marks = $mark_storage->loadMultiple();

      foreach ($marks as $mark) {
        $this->marks[$mark->id()] = $mark;
      }
    }
    catch (\Exception $e) {
      // Log error but continue.
      \Drupal::logger('prosemirror')->error('Failed to load marks: @error', [
        '@error' => $e->getMessage(),
      ]);
    }

    return $this->marks;
  }

  /**
   * Gets only configurable elements.
   *
   * @return \Drupal\prosemirror\ProseMirrorElementInterface[]
   *   Array of configurable elements keyed by machine name.
   */
  public function getConfigurableElements(): array {
    $elements = [];

    try {
      $element_storage = $this->entityTypeManager->getStorage('prosemirror_element');
      $elements = $element_storage->loadMultiple();
    }
    catch (\Exception $e) {
      \Drupal::logger('prosemirror')->error('Failed to load configurable elements: @error', [
        '@error' => $e->getMessage(),
      ]);
    }

    return $elements;
  }

  /**
   * Gets only system elements.
   *
   * @return \Drupal\prosemirror\ProseMirrorElementInterface[]
   *   Array of system elements keyed by machine name.
   */
  public function getSystemElements(): array {
    $elements = [];
    $system_definitions = SystemElementTypes::getDefinitions();

    foreach ($system_definitions as $machine_name => $definition) {
      $elements[$machine_name] = new SystemElement([
        'id' => $machine_name,
        'definition' => $definition,
      ], 'prosemirror_element');
    }

    return $elements;
  }

  /**
   * Checks if an element exists.
   *
   * @param string $machineName
   *   The machine name.
   *
   * @return bool
   *   TRUE if the element exists, FALSE otherwise.
   */
  public function hasElement(string $machineName): bool {
    $elements = $this->getAllElements();
    return isset($elements[$machineName]);
  }

  /**
   * Checks if a mark exists.
   *
   * @param string $machineName
   *   The machine name.
   *
   * @return bool
   *   TRUE if the mark exists, FALSE otherwise.
   */
  public function hasMark(string $machineName): bool {
    $marks = $this->getAllMarks();
    return isset($marks[$machineName]);
  }

  /**
   * Clears the element and mark cache.
   */
  public function clearCache(): void {
    $this->elements = NULL;
    $this->marks = NULL;
  }

}
