<?php

declare(strict_types=1);

namespace Drupal\prosemirror\Element;

use Drupal\prosemirror\ProseMirrorElementInterface;
use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Virtual entity for system element types.
 *
 * This class represents runtime-only element types that are not stored in
 * configuration but can still behave like regular ProseMirror elements.
 */
class SystemElement extends ConfigEntityBase implements ProseMirrorElementInterface {

  /**
   * The element definition.
   *
   * @var array
   */
  protected array $definition;

  /**
   * Constructs a SystemElement.
   *
   * @param array $values
   *   An array of values to set, keyed by property name.
   * @param string $entity_type
   *   The entity type ID.
   */
  public function __construct(array $values, $entity_type) {
    // Extract the definition from values.
    $this->definition = $values['definition'] ?? [];
    unset($values['definition']);

    // Set standard properties.
    $values['id'] = $values['id'] ?? $values['machine_name'] ?? '';
    $values['label'] = $this->definition['label'] ?? '';
    $values['machine_name'] = $values['id'];

    parent::__construct($values, $entity_type);
  }

  /**
   * {@inheritdoc}
   */
  public function getType() {
    return $this->definition['type'] ?? 'base_node';
  }

  /**
   * {@inheritdoc}
   */
  public function getCssClasses() {
    return $this->definition['css_classes'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getContent() {
    return $this->definition['content'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getContentMin() {
    return $this->definition['content_min'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getContentMax() {
    return $this->definition['content_max'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getGroups() {
    return $this->definition['groups'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getOptions() {
    return ($this->definition['options'] ?? []) + [
      'element_id' => $this->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getContentExpression() {
    return $this->definition['content_expression'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function isNew() {
    // System elements are never "new" in the entity sense.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function save() {
    // System elements cannot be saved.
    throw new \LogicException('System elements cannot be saved.');
  }

  /**
   * {@inheritdoc}
   */
  public function delete() {
    // System elements cannot be deleted.
    throw new \LogicException('System elements cannot be deleted.');
  }

  /**
   * Checks if this is a system element.
   *
   * @return bool
   *   Always returns TRUE for system elements.
   */
  public function isSystem(): bool {
    return TRUE;
  }

}
