<?php

declare(strict_types=1);

namespace Drupal\prosemirror\Element;

/**
 * Defines system element types that are runtime-only and not configurable.
 *
 * Some of the types are strictly defined by the ProseMirror schema like `doc` and `text`.
 * Others are defined by the ProseMirror module like `media`, `code_block`, `table`, `bullet_list`, `ordered_list`, `heading_*`
 * as they require special handling.
 */
class SystemElementTypes {

  /**
   * Gets all system element type definitions.
   *
   * @return array
   *   Array of element definitions keyed by machine name.
   */
  public static function getDefinitions(): array {
    // Get global settings.
    $config = \Drupal::config('prosemirror.settings');

    return [
      'doc' => [
        'label' => 'Document',
        'type' => 'system',
        'groups' => ['root'],
        'content' => 'block',
        'content_expression' => '+',
        'system' => TRUE,
      ],
      'text' => [
        'label' => 'Text',
        'type' => 'system',
        'groups' => ['inline'],
        'content' => '',
        'content_expression' => '',
        'system' => TRUE,
      ],
      'media' => [
        'label' => 'Media',
        'type' => 'system',
        'groups' => ['inline'],
        'content' => '',
        'content_expression' => '',
        'system' => TRUE,
        'enabled' => TRUE,
      ],
      'code_block' => [
        'label' => 'Code Block',
        'type' => 'system',
        'groups' => ['block'],
        'content' => '',
        'content_expression' => '',
        'system' => TRUE,
        'enabled' => TRUE,
        'languages' => $config->get('code_block_languages') ?? _prosemirror_get_default_code_block_languages(),
      ],
      'table' => [
        'label' => 'Table',
        'type' => 'system',
        'groups' => ['block'],
        'content' => 'table_row',
        'content_expression' => '+',
        'system' => TRUE,
        'enabled' => TRUE,
        'max_rows' => $config->get('table_max_rows') ?? 50,
        'max_columns' => $config->get('table_max_columns') ?? 10,
        'allow_header_rows' => $config->get('table_allow_header_rows') ?? TRUE,
      ],
      'table_row' => [
        'label' => 'Table Row',
        'type' => 'system',
        'groups' => [],
        'content' => 'table_cell table_header',
        'content_expression' => '+',
        'system' => TRUE,
      ],
      'table_cell' => [
        'label' => 'Table Cell',
        'type' => 'system',
        'groups' => [],
        'content' => 'block',
        'content_expression' => '+',
        'system' => TRUE,
      ],
      'table_header' => [
        'label' => 'Table Header',
        'type' => 'system',
        'groups' => [],
        'content' => 'block',
        'content_expression' => '+',
        'system' => TRUE,
      ],
      'bullet_list' => [
        'label' => 'Bullet List',
        'type' => 'system',
        'groups' => ['block'],
        'content' => 'list_item',
        'content_expression' => '+',
        'system' => TRUE,
        'enabled' => TRUE,
      ],
      'ordered_list' => [
        'label' => 'Ordered List',
        'type' => 'system',
        'groups' => ['block'],
        'content' => 'list_item',
        'content_expression' => '+',
        'system' => TRUE,
        'enabled' => TRUE,
      ],
      'list_item' => [
        'label' => 'List Item',
        'type' => 'system',
        'groups' => [],
        'content' => 'block',
        'content_expression' => '+',
        'system' => TRUE,
      ],
      // Add headline types.
      'heading' => [
        'label' => 'Heading',
        'type' => 'system',
        'groups' => ['block'],
        'content' => 'inline',
        'content_expression' => '*',
        'system' => TRUE,
        'enabled' => TRUE,
        'levels' => static::getAvailableHeadingLevels(),
      ],
    ];
  }

  /**
   * Checks if a given machine name is a system type.
   *
   * @param string $machineName
   *   The machine name to check.
   *
   * @return bool
   *   TRUE if it's a system type, FALSE otherwise.
   */
  public static function isSystemType(string $machineName): bool {
    $definitions = static::getDefinitions();
    return isset($definitions[$machineName]);
  }

  /**
   * Gets available heading levels from configured heading elements.
   *
   * @return array
   *   Array of available heading levels.
   */
  protected static function getAvailableHeadingLevels(): array {
    $levels = [];

    try {
      $element_storage = \Drupal::entityTypeManager()->getStorage('prosemirror_element');
      $elements = $element_storage->loadMultiple();

      foreach ($elements as $element) {
        // Check if this is a heading element.
        if ($element->getType() === 'base_node') {
          $options = $element->getOptions();
          if (isset($options['level']) && is_numeric($options['level'])) {
            $level = (int) $options['level'];
            if ($level >= 1 && $level <= 6) {
              $levels[] = $level;
            }
          }
        }
      }
    }
    catch (\Exception $e) {
      // If we can't load elements (e.g., during installation), default to all levels.
      $levels = [1, 2, 3, 4, 5, 6];
    }

    // Sort and remove duplicates.
    $levels = array_unique($levels);
    sort($levels);

    // If no heading elements found, default to basic levels.
    if (empty($levels)) {
      $levels = [1, 2, 3, 4, 5, 6];
    }

    return $levels;
  }

}
