<?php

namespace Drupal\prosemirror\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for the ProseMirror element add/edit forms.
 */
class ProseMirrorElementForm extends EntityForm {

  /**
   * The ProseMirror element type plugin manager.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager
   */
  protected $elementTypeManager;

  /**
   * Constructs a ProseMirrorElementForm object.
   *
   * @param \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager $element_type_manager
   *   The ProseMirror element type plugin manager.
   */
  public function __construct(ProseMirrorElementTypeManager $element_type_manager) {
    $this->elementTypeManager = $element_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.prosemirror_element_type')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\prosemirror\ProseMirrorElementInterface $element */
    $element = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $element->label(),
      '#description' => $this->t('Name of the ProseMirror element.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $element->id(),
      '#machine_name' => [
        'exists' => '\Drupal\prosemirror\Entity\ProseMirrorElement::load',
      ],
      '#disabled' => !$element->isNew(),
    ];

    // Element type selection.
    $type_options = [];
    foreach ($this->elementTypeManager->getDefinitions() as $plugin_id => $definition) {
      $type_options[$plugin_id] = $definition['label'];
    }

    $form['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Element Type'),
      '#options' => $type_options,
      '#default_value' => $element->getType(),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateTypeOptions',
        'wrapper' => 'type-options-wrapper',
      ],
    ];

    // CSS classes.
    $form['css_classes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CSS Classes'),
      '#default_value' => implode(' ', $element->getCssClasses() ?: []),
      '#description' => $this->t('Space-separated list of CSS classes.'),
    ];

    // Content reference.
    $content_options = $this->getContentOptions($element);

    $form['content'] = [
      '#type' => 'select',
      '#title' => $this->t('Content'),
      '#default_value' => $element->getContent(),
      '#options' => $content_options,
      '#description' => $this->t('Select an element or group that defines what content this element can contain.'),
      '#required' => TRUE,
      '#empty_option' => $this->t('- Select content -'),
    ];

    // Content min/max.
    $form['content_limits'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content Limits'),
    ];

    $form['content_limits']['content_min'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum'),
      '#default_value' => $element->getContentMin(),
      '#min' => 0,
    ];

    $form['content_limits']['content_max'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum'),
      '#default_value' => $element->getContentMax(),
      '#min' => 0,
    ];

    // Groups.
    $group_options = [];
    $groups = $this->entityTypeManager->getStorage('prosemirror_element_group')->loadMultiple();
    foreach ($groups as $group) {
      $group_options[$group->id()] = $group->label();
    }

    $form['groups'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Groups'),
      '#options' => $group_options,
      '#default_value' => $element->getGroups() ?: [],
      '#description' => $this->t('Select the groups this element belongs to.'),
    ];

    // Type-specific options.
    $form['type_options'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'type-options-wrapper'],
      '#tree' => TRUE,
    ];

    if ($element->getType()) {
      try {
        /** @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeInterface $plugin */
        $plugin = $this->elementTypeManager->createInstance($element->getType());
        $form['type_options'] = $plugin->buildOptionsForm($form['type_options'], $form_state, $element);
      }
      catch (\Exception $e) {
        // Handle plugin not found.
      }
    }

    return $form;
  }

  /**
   * Maps option types to form element types.
   */
  protected function getFormElementType($type) {
    switch ($type) {
      case 'boolean':
        return 'checkbox';

      case 'integer':
        return 'number';

      case 'array':
        return 'textarea';

      default:
        return 'textfield';
    }
  }

  /**
   * Gets content options for the select field.
   *
   * @param \Drupal\prosemirror\ProseMirrorElementInterface $current_element
   *   The current element being edited.
   *
   * @return array
   *   An array of options grouped by type.
   */
  protected function getContentOptions($current_element) {
    $options = [];

    // Add built-in types.
    $options[$this->t('Built-in Types')->__toString()] = [
      'text' => $this->t('Text'),
    ];

    // Add groups.
    $groups = $this->entityTypeManager->getStorage('prosemirror_element_group')->loadMultiple();
    if ($groups) {
      $group_options = [];
      foreach ($groups as $group) {
        $group_options[$group->id()] = $group->label();
      }
      asort($group_options);
      $options[$this->t('Groups')->__toString()] = $group_options;
    }

    // Add elements (exclude current element to prevent circular reference).
    $elements = $this->entityTypeManager->getStorage('prosemirror_element')->loadMultiple();
    if ($elements) {
      $element_options = [];
      foreach ($elements as $element) {
        // Exclude the current element.
        if ($current_element->id() && $element->id() === $current_element->id()) {
          continue;
        }
        $element_options[$element->id()] = $element->label();
      }
      asort($element_options);
      $options[$this->t('Elements')->__toString()] = $element_options;
    }

    return $options;
  }

  /**
   * AJAX callback to update type-specific options.
   */
  public function updateTypeOptions(array &$form, FormStateInterface $form_state) {
    // Clear the wrapper first.
    $form['type_options'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'type-options-wrapper'],
      '#tree' => TRUE,
    ];

    // Get the selected type and rebuild options.
    $type = $form_state->getValue('type');
    if ($type) {
      try {
        /** @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeInterface $plugin */
        $plugin = $this->elementTypeManager->createInstance($type);
        // Create a temporary element to pass to the plugin.
        $temp_element = $this->entity->createDuplicate();
        $temp_element->set('type', $type);
        $form['type_options'] = $plugin->buildOptionsForm($form['type_options'], $form_state, $temp_element);
      }
      catch (\Exception $e) {
        // Handle plugin not found.
      }
    }

    return $form['type_options'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    /** @var \Drupal\prosemirror\ProseMirrorElementInterface $element */
    $element = $this->entity;

    // Validate CSS classes.
    $css_classes = $form_state->getValue('css_classes');
    if (!empty($css_classes)) {
      $classes = array_filter(array_map('trim', explode(' ', $css_classes)));
      foreach ($classes as $class) {
        if (!preg_match('/^[a-zA-Z_][\w-]*$/', $class)) {
          $form_state->setErrorByName('css_classes', $this->t('CSS class "@class" is not valid. CSS classes must start with a letter or underscore and contain only letters, numbers, hyphens, and underscores.', ['@class' => $class]));
          break;
        }
      }
    }

    // Content reference validation is not needed anymore since we use a select field with valid options.

    // Validate content limits.
    $content_min = $form_state->getValue(['content_limits', 'content_min']);
    $content_max = $form_state->getValue(['content_limits', 'content_max']);

    if ($content_min !== '' && $content_min !== NULL && $content_max !== '' && $content_max !== NULL) {
      if ($content_min > $content_max) {
        $form_state->setErrorByName('content_limits][content_max', $this->t('Maximum must be greater than or equal to minimum.'));
      }
    }

    // Validate type-specific options using the plugin.
    $type = $form_state->getValue('type') ?: $element->getType();
    if ($type) {
      try {
        /** @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeInterface $plugin */
        $plugin = $this->elementTypeManager->createInstance($type);
        $plugin->validateOptionsForm($form['type_options'], $form_state);
      }
      catch (\Exception $e) {
        $form_state->setErrorByName('type', $this->t('Invalid element type selected.'));
      }
    }

    // Validate that element doesn't create circular references.
    $content = $form_state->getValue('content');

    if ($element->id() && $content === $element->id()) {
      $form_state->setErrorByName('content', $this->t('An element cannot reference itself as content.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $element = $this->entity;

    // Process CSS classes.
    $css_classes = array_filter(array_map('trim', explode(' ', $form_state->getValue('css_classes'))));
    $element->set('css_classes', $css_classes);

    // Process content limits.
    $element->set('content_min', $form_state->getValue(['content_limits', 'content_min']));
    $element->set('content_max', $form_state->getValue(['content_limits', 'content_max']));

    // Process groups.
    $groups = array_filter($form_state->getValue('groups'));
    $element->set('groups', array_values($groups));

    // Process type-specific options using the plugin.
    if ($element->getType()) {
      try {
        /** @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeInterface $plugin */
        $plugin = $this->elementTypeManager->createInstance($element->getType());
        $type_options = $form_state->getValue('type_options', []);
        $processed_options = $plugin->processFormValues($type_options);
        $element->set('options', $processed_options);
      }
      catch (\Exception $e) {
        // If plugin not found, save raw values.
        $element->set('options', $form_state->getValue('type_options', []));
      }
    }

    $status = $element->save();

    switch ($status) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('Created the %label ProseMirror element.', [
          '%label' => $element->label(),
        ]));
        break;

      default:
        $this->messenger()->addStatus($this->t('Saved the %label ProseMirror element.', [
          '%label' => $element->label(),
        ]));
    }
    $form_state->setRedirectUrl($element->toUrl('collection'));
  }

}
