<?php

declare(strict_types=1);

namespace Drupal\prosemirror\Plugin\Editor;

use Drupal\prosemirror\Element\SystemElementTypes;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\editor\Attribute\Editor;
use Drupal\editor\Plugin\EditorBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\editor\Entity\Editor as EditorEntity;
use Psr\Log\LoggerInterface;
use Drupal\Core\Url;
use Drupal\media_library\MediaLibraryState;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Drupal\prosemirror\Plugin\ProseMirrorExtensionManager;

/**
 * Defines a ProseMirror-based text editor for Drupal.
 *
 * @internal
 *   Plugin classes are internal.
 */
#[Editor(
  id: 'prosemirror',
  label: new TranslatableMarkup('ProseMirror'),
  supports_content_filtering: TRUE,
  supports_inline_editing: TRUE,
  is_xss_safe: FALSE,
  supported_element_types: [
    'textarea',
  ]
)]
class ProseMirror extends EditorBase implements ContainerFactoryPluginInterface {

  /**
   * Gets the given URL with all placeholders replaced.
   *
   * @param \Drupal\Core\Url $url
   *   A URL which generates CSRF token placeholders.
   *
   * @return string
   *   The URL string, with all placeholders replaced.
   */
  private static function getUrlWithReplacedCsrfTokenPlaceholder(Url $url): string {
    $generated_url = $url->toString(TRUE);
    $url_with_csrf_token_placeholder = [
      '#plain_text' => $generated_url->getGeneratedUrl(),
    ];
    $generated_url->applyTo($url_with_csrf_token_placeholder);
    return (string) \Drupal::service('renderer')->renderInIsolation($url_with_csrf_token_placeholder);
  }

  /**
   * A logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The media type entity storage.
   *
   * @var \Drupal\Core\Config\Entity\ConfigEntityStorageInterface
   */
  protected $mediaTypeStorage;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The ProseMirror element type plugin manager.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager
   */
  protected $elementTypeManager;

  /**
   * The ProseMirror extension manager.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirrorExtensionManager
   */
  protected $extensionManager;

  /**
   * Constructs an editor plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager $element_type_manager
   *   The ProseMirror element type plugin manager.
   * @param \Drupal\prosemirror\Plugin\ProseMirrorExtensionManager $extension_manager
   *   The ProseMirror extension manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, LoggerInterface $logger, EntityTypeManagerInterface $entity_type_manager, ProseMirrorElementTypeManager $element_type_manager, ProseMirrorExtensionManager $extension_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->logger = $logger;
    $this->entityTypeManager = $entity_type_manager;
    $this->mediaTypeStorage = $entity_type_manager->getStorage('media_type');
    $this->elementTypeManager = $element_type_manager;
    $this->extensionManager = $extension_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.channel.prosemirror'),
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.prosemirror_element_type'),
      $container->get('plugin.manager.prosemirror_extension')
    );
  }

  /**
   * Gets the media settings for the editor.
   *
   * @param \Drupal\editor\Entity\Editor $editor
   *   The editor entity.
   *
   * @return array
   *   The media settings array.
   */
  protected function getMediaSettings(EditorEntity $editor) {
    $media_type_ids = $this->mediaTypeStorage->getQuery()->execute();

    // Making the title for editor drupal media embed translatable.
    $static_plugin_config['drupalMedia']['dialogSettings']['title'] = $this->t('Add or select media');

    if ($editor->hasAssociatedFilterFormat()) {
      $media_embed_filter = $editor->getFilterFormat()->filters()->get('media_embed');
      // Optionally limit the allowed media types based on the MediaEmbed
      // setting. If the setting is empty, do not limit the options.
      if (!empty($media_embed_filter->settings['allowed_media_types'])) {
        $media_type_ids = array_intersect_key($media_type_ids, $media_embed_filter->settings['allowed_media_types']);
      }
    }

    $state = MediaLibraryState::create(
      'media_library.opener.editor',
      $media_type_ids,
      in_array('image', $media_type_ids) ? 'image' : reset($media_type_ids),
      1,
      ['filter_format_id' => $editor->getFilterFormat()->id()],
    );

    $library_url = Url::fromRoute('media_library.ui')
      ->setOption('query', $state->all())
      ->toString(TRUE)
      ->getGeneratedUrl();

    $dynamic_plugin_config = $static_plugin_config;
    $dynamic_plugin_config['drupalMedia']['libraryURL'] = $library_url;

    $dynamic_plugin_config['drupalMedia']['previewURL'] = Url::fromRoute('media.filter.preview')
      ->setRouteParameter('filter_format', $editor->getFilterFormat()->id())
      ->toString(TRUE)
      ->getGeneratedUrl();

    $dynamic_plugin_config['drupalMedia']['metadataUrl'] = self::getUrlWithReplacedCsrfTokenPlaceholder(
      Url::fromRoute('prosemirror.media_entity_metadata')
        ->setRouteParameter('editor', $editor->id())
    );
    $dynamic_plugin_config['drupalMedia']['previewCsrfToken'] = \Drupal::csrfToken()->get('X-Drupal-MediaPreview-CSRF-Token');

    return $dynamic_plugin_config;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntitySelectorSettings(EditorEntity $editor): array {
    if(!\Drupal::service('module_handler')->moduleExists('linkit')) {
      return ['profiles'=>[]];
    }

    // Load all linkit_profile profiles.
    $linkit_profiles = \Drupal::entityTypeManager()->getStorage('linkit_profile')
      ->loadMultiple();
    $profile_ids = array_keys($linkit_profiles);

    $profiles = [];

    if(in_array('prosemirror_content', $profile_ids)) {
      $profiles['node'] = [
        'profile' => 'prosemirror_content',
        'autocompleteUrl' => Url::fromRoute('linkit.autocomplete', ['linkit_profile_id' => 'prosemirror_content'])
          ->toString(TRUE)
          ->getGeneratedUrl(),
        'label' => $this->t('Content')
      ];
    }
    if(in_array('prosemirror_block_content', $profile_ids)) {
      $profiles['block'] = [
          'profile' => 'prosemirror_block_content',
          'autocompleteUrl' => Url::fromRoute('linkit.autocomplete', ['linkit_profile_id' => 'prosemirror_block_content'])
            ->toString(TRUE)
            ->getGeneratedUrl(),
          'label' => $this->t('Block content')
        ];
    }
    if(in_array('prosemirror_media', $profile_ids)) {
      $profiles['media'] = [
          'profile' => 'prosemirror_media',
          'autocompleteUrl' => Url::fromRoute('linkit.autocomplete', ['linkit_profile_id' => 'prosemirror_media'])
            ->toString(TRUE)
            ->getGeneratedUrl(),
          'label' => $this->t('Media')
        ];
    }

    return [
      'profiles' => $profiles,
    ];
  }

  /**
   * Gets the components configuration for ProseMirror.
   *
   * @param \Drupal\editor\Entity\Editor $editor
   *   The editor entity.
   *
   * @return array
   *   The components configuration array.
   */
  public function getComponentsSettings(EditorEntity $editor): array {
    $components = [];

    // Load all ProseMirror elements.
    $element_storage = $this->entityTypeManager->getStorage('prosemirror_element');
    $elements = $element_storage->loadMultiple();

    /** @var \Drupal\prosemirror\ProseMirrorElementInterface $element */
    foreach ($elements as $element) {
      try {
        /** @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeInterface $plugin */
        $plugin = $this->elementTypeManager->createInstance($element->getType(), $element->getOptions());
        $components[$element->id()] = $plugin->buildJavaScriptConfiguration($element);
      }
      catch (\Exception $e) {
        // Log error if plugin not found.
        $this->logger->error('Could not load element type plugin @type for element @id', [
          '@type' => $element->getType(),
          '@id' => $element->id(),
        ]);
      }
    }

    return $components;
  }

  /**
   * Gets the system nodes configuration for ProseMirror.
   *
   * @param \Drupal\editor\Entity\Editor $editor
   *   The editor entity.
   *
   * @return array
   *   The system nodes configuration array.
   */
  protected function getSystemNodesSettings(EditorEntity $editor): array {
    // Get filter settings if available.
    $filter_settings = $this->getFilterSettings($editor);
    $system_elements = $filter_settings['system_elements'] ?? [];

    // Get the configured groups and default element from global settings.
    $config = \Drupal::config('prosemirror.settings');
    $root_group = $config->get('root_group') ?? 'block';
    $text_group = $config->get('text_group') ?? 'inline';
    $default_element = $config->get('default_element') ?? 'paragraph';

    // Create doc content expression using the configured root group.
    $doc_content = $root_group . '+';

    $system_nodes = [
      'doc' => [
        'content' => $doc_content,
      ],
      'text' => [
        'group' => $text_group,
      ],
      'defaultElement' => $default_element,
    ];

    // Check if any system elements are actually enabled (not just if the array exists)
    $any_elements_enabled = !empty(array_filter($system_elements));

    // Add configurable system elements with their settings from global definitions.
    $system_definitions = SystemElementTypes::getDefinitions();

    foreach (['media', 'code_block', 'table', 'bullet_list', 'ordered_list', 'heading'] as $element_id) {
      if (isset($system_definitions[$element_id])) {
        // If no elements are actually enabled, include all elements (default Drupal behavior)
        // If elements are enabled, only include the enabled ones.
        $element_enabled = !$any_elements_enabled || !empty($system_elements[$element_id]);

        if ($element_enabled) {
          // Use full configuration from global system definitions.
          $system_nodes[$element_id] = $system_definitions[$element_id];
        }
      }
    }

    return $system_nodes;
  }

  /**
   * Gets the marks configuration for ProseMirror.
   *
   * @param \Drupal\editor\Entity\Editor $editor
   *   The editor entity.
   *
   * @return array
   *   The marks configuration array.
   */
  protected function getMarksSettings(EditorEntity $editor): array {
    $marks = [];

    // Get filter settings if available.
    $filter_settings = $this->getFilterSettings($editor);
    $marks_settings = $filter_settings['marks'] ?? [];

    // Check if any marks are actually enabled (not just if the array exists)
    $any_marks_enabled = !empty(array_filter($marks_settings));

    // Load all ProseMirror marks.
    $mark_storage = $this->entityTypeManager->getStorage('prosemirror_mark');
    $mark_entities = $mark_storage->loadMultiple();

    /** @var \Drupal\prosemirror\ProseMirrorMarkInterface $mark */
    foreach ($mark_entities as $mark) {
      $mark_id = $mark->id();

      // If no marks are actually enabled, include all marks (default Drupal behavior)
      // If marks are enabled, only include the enabled ones.
      $mark_enabled = !$any_marks_enabled || !empty($marks_settings[$mark_id]);

      if ($mark_enabled) {
        $mark_config = [
          'parse_dom' => $mark->getParseDom(),
          'to_dom' => $mark->getToDom(),
          'attributes' => $mark->getAttributes(),
        ];

        // Add variants if available.
        $variants = $mark->getVariants();
        if (!empty($variants)) {
          $mark_config['variants'] = $variants;
        }

        $marks[$mark_id] = $mark_config;
      }
    }

    return $marks;
  }

  /**
   * Gets the filter settings for the editor's text format.
   *
   * @param \Drupal\editor\Entity\Editor $editor
   *   The editor entity.
   *
   * @return array
   *   The filter settings array.
   */
  protected function getFilterSettings(EditorEntity $editor): array {
    if (!$editor->hasAssociatedFilterFormat()) {
      return [];
    }

    $format = $editor->getFilterFormat();
    $prosemirror_filter = $format->filters()->get('prosemirror_filter');

    if (!$prosemirror_filter || !$prosemirror_filter->status) {
      return [];
    }

    return $prosemirror_filter->getConfiguration()['settings'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getJSSettings(EditorEntity $editor) {
    $settings = [];

    $settings['config']['licenseKey'] ??= 'MIT';

    $settings['config']['plugins']['media'] = $this->getMediaSettings($editor);

    $settings['config']['plugins']['entitySelector'] = $this->getEntitySelectorSettings($editor);

    $settings['config']['components'] = $this->getComponentsSettings($editor);

    $settings['config']['systemNodes'] = $this->getSystemNodesSettings($editor);

    $settings['config']['marks'] = $this->getMarksSettings($editor);

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function getLibraries(EditorEntity $editor) {
    $plugin_libraries = [];

    $plugin_libraries[] = 'prosemirror/editor';

    // Add libraries from registered extensions.
    $extension_libraries = $this->extensionManager->getExtensionLibraries();
    $plugin_libraries = array_merge($plugin_libraries, $extension_libraries);

    return $plugin_libraries;
  }

}
