<?php

namespace Drupal\prosemirror\Plugin\Filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\prosemirror\Rendering\ProseMirrorRenderer;
use Drupal\prosemirror\Element\SystemElementTypes;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a filter for rendering ProseMirror content.
 *
 * @Filter(
 *   id = "prosemirror_filter",
 *   title = @Translation("ProseMirror Content Renderer"),
 *   description = @Translation("Renders ProseMirror JSON content as HTML."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_MARKUP_LANGUAGE,
 *   weight = -50
 * )
 */
class ProseMirrorFilter extends FilterBase implements ContainerFactoryPluginInterface {

  /**
   * The ProseMirror renderer service.
   *
   * @var \Drupal\prosemirror\Rendering\ProseMirrorRenderer
   */
  protected ProseMirrorRenderer $renderer;

  /**
   * Constructs a ProseMirrorFilter object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\prosemirror\Rendering\ProseMirrorRenderer $renderer
   *   The ProseMirror renderer service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ProseMirrorRenderer $renderer) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('prosemirror.renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    // Try to decode JSON content.
    $data = json_decode($text, TRUE);

    // If it's not valid JSON, pass through unchanged. Will be sanitized by Drupal.
    if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
      return new FilterProcessResult($text);
    }

    // Render the ProseMirror content.
    $rendered = $this->renderer->render($data);

    return new FilterProcessResult($rendered);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $settings = $this->settings;

    // Get all available element types (system + configured)
    $element_options = [];

    // Get system elements (except doc and text)
    $system_elements = SystemElementTypes::getDefinitions();
    foreach ($system_elements as $id => $definition) {
      // Skip internal structure elements.
      $internal_elements = ['doc', 'text', 'table_row', 'table_cell', 'table_header', 'list_item'];
      if (!in_array($id, $internal_elements) && ($definition['enabled'] ?? FALSE)) {
        $element_options[$id] = $definition['label'];
      }
    }

    // Get configured elements.
    $element_storage = \Drupal::entityTypeManager()->getStorage('prosemirror_element');
    $element_entities = $element_storage->loadMultiple();
    foreach ($element_entities as $element) {
      $element_options[$element->id()] = $element->label();
    }

    // Extract selected system elements - handle both old complex format and new simple format.
    $selected_system_elements = [];
    if (isset($settings['system_elements'])) {
      if (is_array($settings['system_elements'])) {
        // If it's already a simple array of selected keys, use it.
        $selected_system_elements = array_keys(array_filter($settings['system_elements']));
      }
    }

    $form['system_elements'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed Elements'),
      '#description' => $this->t('Select which elements are allowed in this text format. If none are selected, all are allowed.'),
      '#options' => $element_options,
      '#default_value' => $selected_system_elements,
    ];

    // Get available marks from mark entities.
    $mark_storage = \Drupal::entityTypeManager()->getStorage('prosemirror_mark');
    $mark_entities = $mark_storage->loadMultiple();

    $mark_options = [];
    foreach ($mark_entities as $mark) {
      $mark_options[$mark->id()] = $mark->label();
    }

    // Extract selected marks - handle both old complex format and new simple format.
    $selected_marks = [];
    if (isset($settings['marks'])) {
      if (is_array($settings['marks'])) {
        // If it's already a simple array of selected keys, use it.
        $selected_marks = array_keys(array_filter($settings['marks']));
      }
    }

    $form['marks'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed Text Formatting Marks'),
      '#description' => $this->t('Select which text formatting marks are allowed in this text format. If none are selected, all are allowed.'),
      '#options' => $mark_options,
      '#default_value' => $selected_marks,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    // Filter out unchecked values (Drupal checkboxes return 0 for unchecked)
    $system_elements = array_filter($form_state->getValue('system_elements'));
    $marks = array_filter($form_state->getValue('marks'));

    $this->configuration['settings']['system_elements'] = $system_elements;
    $this->configuration['settings']['marks'] = $marks;
  }

  /**
   * Gets the filter settings.
   *
   * @return array
   *   The filter settings.
   */
  public function getFilterSettings(): array {
    return $this->settings;
  }

  /**
   * {@inheritdoc}
   */
  public function tips($long = FALSE) {
    if ($long) {
      return $this->t('ProseMirror JSON content will be automatically rendered as HTML.');
    }
    return $this->t('ProseMirror content is rendered as HTML.');
  }

}
