<?php

namespace Drupal\prosemirror\Plugin\ProseMirror\ElementType;

use Drupal\prosemirror\Plugin\ProseMirrorElementTypeBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\prosemirror\ProseMirrorElementInterface;
use Drupal\prosemirror\Transformation\ValidationError;

/**
 * Provides a 'leaf_block' element type.
 *
 * @ProseMirrorElementType(
 *   id = "leaf_block",
 *   label = @Translation("Leaf Block"),
 *   description = @Translation("A block element that contains only inline content.")
 * )
 */
class LeafBlock extends ProseMirrorElementTypeBase {

  /**
   * {@inheritdoc}
   */
  public function getAvailableOptions() {
    return [
      // E.g., media or <hr/> are not editable as they cannot contain content.
      'editable' => [
        'label' => $this->t('Editable'),
        'type' => 'boolean',
        'default' => FALSE,
        'required' => FALSE,
        'description' => $this->t('Whether the content of this block is editable.'),
      ],
      // E.g., to render some links as `primary buttons`, `secondary buttons`, or regular `links`.
      'variants' => [
        'label' => $this->t('Variants'),
        'type' => 'array',
        'default' => [],
        'required' => FALSE,
        'description' => $this->t('Available style variants for this block.'),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(array $form, FormStateInterface $form_state, ProseMirrorElementInterface $element) {
    $options = $element->getOptions();

    // Editable field.
    $form['editable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Editable'),
      '#default_value' => $options['editable'] ?? FALSE,
      '#description' => $this->t('Whether the content of this block is editable.'),
    ];

    // Variants field.
    $form['variants'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Variants'),
      '#default_value' => $this->variantsToString($options['variants'] ?? []),
      '#description' => $this->t('Style variants allow users to apply different visual styles to this block. Enter one variant per line in the format: <code>key|label</code><br><br><strong>Examples:</strong><br><code>highlight|Highlighted Text</code><br><code>warning|Warning Message</code><br><code>info|Information Box</code><br><code>success|Success Message</code><br><br>The key must contain only lowercase letters, numbers, and underscores. The label is what users will see in the editor.'),
      '#rows' => 5,
      '#attributes' => [
        'placeholder' => "highlight|Highlighted Text\nwarning|Warning Message\ninfo|Information Box",
      ],
    ];

    return $form;
  }

  /**
   * Converts variants array to string format for textarea.
   */
  protected function variantsToString(array $variants) {
    $lines = [];
    foreach ($variants as $key => $label) {
      $lines[] = $key . '|' . $label;
    }
    return implode("\n", $lines);
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue('type_options');

    // Validate variants.
    if (!empty($values['variants'])) {
      $lines = array_filter(array_map('trim', explode("\n", $values['variants'])));
      foreach ($lines as $line) {
        if (strpos($line, '|') === FALSE) {
          $form_state->setErrorByName('type_options][variants', $this->t('Each variant must be in the format: key|label'));
          break;
        }

        [$key, $label] = array_map('trim', explode('|', $line, 2));

        if (!preg_match('/^[a-z0-9_]+$/', $key)) {
          $form_state->setErrorByName('type_options][variants', $this->t('Variant key "@key" must contain only lowercase letters, numbers, and underscores.', ['@key' => $key]));
          break;
        }

        if (empty($label)) {
          $form_state->setErrorByName('type_options][variants', $this->t('Variant label cannot be empty for key "@key".', ['@key' => $key]));
          break;
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function processFormValues(array $values) {
    // Process editable.
    $values['editable'] = !empty($values['editable']);

    // Process variants.
    if (!empty($values['variants'])) {
      $variants = [];
      $lines = array_filter(array_map('trim', explode("\n", $values['variants'])));
      foreach ($lines as $line) {
        if (strpos($line, '|') !== FALSE) {
          [$key, $label] = array_map('trim', explode('|', $line, 2));
          if ($key !== '' && $label !== '') {
            $variants[$key] = $label;
          }
        }
      }
      $values['variants'] = $variants;
    }
    else {
      $values['variants'] = [];
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptions(array $options) {
    $errors = parent::validateOptions($options);

    // Validate variants.
    if (!empty($options['variants'])) {
      if (!is_array($options['variants'])) {
        $errors['variants'] = $this->t('Variants must be an array.');
      }
      else {
        foreach ($options['variants'] as $key => $label) {
          // Validate variant key.
          if (!preg_match('/^[a-z0-9_]+$/', $key)) {
            $errors['variants'] = $this->t('Variant keys must be valid machine names (lowercase letters, numbers, and underscores only).');
            break;
          }
          // Validate variant label.
          if (!is_string($label) || empty(trim($label))) {
            $errors['variants'] = $this->t('All variant labels must be non-empty strings.');
            break;
          }
        }
      }
    }

    return $errors;
  }

  /**
   * Get allowed variants based on the element configuration.
   */
  protected function getAllowedVariants() {
    return $this->getOptionsConfig()['variants'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function validateAttributes(array $attrs, array $path, array &$errors, array &$references): array {
    $sanitized = [];

    // Validate name attribute.
    if (isset($attrs['name'])) {
      $sanitized['name'] = (string) $attrs['name'];
    }

    // Validate variant attribute.
    if (isset($attrs['variant'])) {
      $variant = (string) $attrs['variant'];

      $allowed_variants = $this->getAllowedVariants();

      if (!array_key_exists($variant, $allowed_variants)) {
        $name = $this->getOptionsConfig()['element_id'] ?? 'element';
        $errors[] = ValidationError::atPath("Invalid variant: '$variant' for '$name'. Allowed variants are: " . implode(', ', array_keys($allowed_variants)), $path);
      }
      else {
        $sanitized['variant'] = $variant;
      }
    }

    return $sanitized;
  }

}
