<?php

namespace Drupal\prosemirror\Plugin\ProseMirror\ElementType;

use Drupal\prosemirror\Plugin\ProseMirrorElementTypeBase;
use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\prosemirror\Transformation\EntityReference;

/**
 * Provides a 'system' element type for built-in ProseMirror system nodes.
 *
 * @ProseMirrorElementType(
 *   id = "system",
 *   label = @Translation("System Element"),
 *   description = @Translation("Built-in system elements like media, icon, tables, etc.")
 * )
 */
class SystemElement extends ProseMirrorElementTypeBase {

  /**
   * {@inheritdoc}
   */
  public function getAvailableOptions() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function validateNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    $type = $node['type'];

    // Handle system types.
    switch ($type) {
      case 'doc':
        return $this->validateDocNode($node, $path, $errors, $references, $transformationHelper);

      case 'text':
        return $this->validateTextNode($node, $path, $errors, $references, $transformationHelper);

      case 'media':
        return $this->validateMediaNode($node, $path, $errors, $references, $transformationHelper);

      case 'code_block':
        return $this->validateCodeBlockNode($node, $path, $errors, $references, $transformationHelper);

      case 'bullet_list':
        return $this->validateBulletListNode($node, $path, $errors, $references, $transformationHelper);

      case 'ordered_list':
        return $this->validateOrderedListNode($node, $path, $errors, $references, $transformationHelper);

      case 'list_item':
        return $this->validateListItemNode($node, $path, $errors, $references, $transformationHelper);

      case 'table':
        return $this->validateTableNode($node, $path, $errors, $references, $transformationHelper);

      case 'table_row':
        return $this->validateTableRowNode($node, $path, $errors, $references, $transformationHelper);

      case 'table_cell':
        return $this->validateTableCellNode($node, $path, $errors, $references, $transformationHelper);

      case 'table_header':
        return $this->validateTableHeaderNode($node, $path, $errors, $references, $transformationHelper);

      default:
        // This should not happen - system element plugin should only handle system types.
        $errors[] = ValidationError::atPath("System element type plugin received non-system type: $type", $path);
        return [];
    }
  }

  /**
   * Validates a simple content-only node (no attributes).
   *
   * @param string $type
   *   The node type.
   * @param array $node
   *   The node data.
   * @param array $path
   *   The path to this node.
   * @param array $errors
   *   Array to collect validation errors.
   * @param array $references
   *   Array to collect entity references.
   * @param mixed $transformationHelper
   *   The transformation helper service.
   *
   * @return array
   *   The sanitized node.
   */
  protected function validateSimpleContentNode(string $type, array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    $sanitized = [
      'type' => $type,
    ];

    // Simple content nodes don't have attributes - strip any that were added.

    // Validate content.
    if (isset($node['content']) && is_array($node['content'])) {
      $sanitized['content'] = [];
      foreach ($node['content'] as $index => $child) {
        $childPath = array_merge($path, ['content', $index]);
        $sanitizedChild = $transformationHelper->validateChildNode($child, $childPath, $errors, $references);
        if (!empty($sanitizedChild)) {
          $sanitized['content'][] = $sanitizedChild;
        }
      }
    }

    return $sanitized;
  }

  /**
   * Validates a doc node.
   */
  protected function validateDocNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateSimpleContentNode('doc', $node, $path, $errors, $references, $transformationHelper);
  }

  /**
   * Validates a node with attributes and content.
   *
   * @param string $type
   *   The node type.
   * @param array $node
   *   The node data.
   * @param array $path
   *   The path to this node.
   * @param array $errors
   *   Array to collect validation errors.
   * @param array $references
   *   Array to collect entity references.
   * @param mixed $transformationHelper
   *   The transformation helper service.
   * @param callable $attributeValidator
   *   Callback to validate attributes.
   *
   * @return array
   *   The sanitized node.
   */
  protected function validateNodeWithAttributes(string $type, array $node, array $path, array &$errors, array &$references, $transformationHelper, callable $attributeValidator): array {
    $sanitized = [
      'type' => $type,
    ];

    // Validate attributes using the provided callback.
    if (isset($node['attrs']) && is_array($node['attrs'])) {
      $sanitized['attrs'] = $attributeValidator($node['attrs'], $path, $errors, $references, $transformationHelper);
    }

    // Validate content if present.
    if (isset($node['content']) && is_array($node['content'])) {
      $sanitized['content'] = [];
      foreach ($node['content'] as $index => $child) {
        $childPath = array_merge($path, ['content', $index]);
        $sanitizedChild = $transformationHelper->validateChildNode($child, $childPath, $errors, $references);
        if (!empty($sanitizedChild)) {
          $sanitized['content'][] = $sanitizedChild;
        }
      }
    }

    return $sanitized;
  }

  /**
   * Validates a text node.
   */
  protected function validateTextNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    $sanitized = [
      'type' => 'text',
    ];

    // Text nodes don't have attributes - strip any that were added.

    // Validate marks.
    if (isset($node['marks']) && is_array($node['marks'])) {
      $sanitized['marks'] = $transformationHelper->sanitizeMarks($node['marks'], $path, $errors, $references);
    }

    // Handle text value.
    if (isset($node['text'])) {
      $sanitized['text'] = (string) $node['text'];
    }

    return $sanitized;
  }

  /**
   * Validates a media node.
   */
  protected function validateMediaNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateNodeWithAttributes('media', $node, $path, $errors, $references, $transformationHelper,
      [$this, 'validateMediaAttributes']);
  }

  /**
   * Validates media node attributes.
   */
  protected function validateMediaAttributes(array $attrs, array $path, array &$errors, array &$references, $transformationHelper): array {
    $sanitized = [];

    // Validate data-entity-type attribute.
    if (isset($attrs['data-entity-type'])) {
      $sanitized['data-entity-type'] = (string) $attrs['data-entity-type'];
    }

    // Validate data-entity-uuid attribute.
    if (isset($attrs['data-entity-uuid'])) {
      $uuid = (string) $attrs['data-entity-uuid'];
      if (!empty($uuid)) {
        $sanitized['data-entity-uuid'] = $uuid;

        $reference = new EntityReference('media', $uuid, $path);

        $entity_repository = $transformationHelper->getEntityRepository();
        if (!$reference->isValid($entity_repository)) {
          $errors[] = ValidationError::atPath('Invalid media selected: media does not exist', $path);
        }

        // Add to references for tracking.
        $references[] = $reference;
      }
    }

    return $sanitized;
  }

  /**
   * Validates code_block node attributes.
   */
  protected function validateCodeBlockAttributes(array $attrs, array $path, array &$errors, array &$references): array {
    $sanitized = [];

    // Validate language attribute, provided as "variant".
    if (!empty($attrs['variant'])) {
      $variant = (string) $attrs['variant'];

      // Load allowed code variants from ProseMirror settings.
      $config = \Drupal::config('prosemirror.settings');
      $allowed_languages = $config->get('code_block_languages') ?? _prosemirror_get_default_code_block_languages();

      // Validate that the selected variant is allowed.
      if (!array_key_exists($variant, $allowed_languages)) {
        $errors[] = ValidationError::atPath("Invalid code block language variant: '$variant'. Allowed variants are: " . implode(', ', array_keys($allowed_languages)), $path);
      }
      else {
        $sanitized['variant'] = $variant;
      }
    }

    // Also allow providing a "name" attribute, which is the name of e.g. the file.
    if (!empty($attrs['name'])) {
      $sanitized['name'] = (string) $attrs['name'];
    }

    return $sanitized;
  }

  /**
   * Validates a code_block node.
   */
  protected function validateCodeBlockNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateNodeWithAttributes('code_block', $node, $path, $errors, $references, $transformationHelper,
      [$this, 'validateCodeBlockAttributes']);
  }

  /**
   * Validates a bullet_list node.
   */
  protected function validateBulletListNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateSimpleContentNode('bullet_list', $node, $path, $errors, $references, $transformationHelper);
  }

  /**
   * Validates an ordered_list node.
   */
  protected function validateOrderedListNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateSimpleContentNode('ordered_list', $node, $path, $errors, $references, $transformationHelper);
  }

  /**
   * Validates a list_item node.
   */
  protected function validateListItemNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateSimpleContentNode('list_item', $node, $path, $errors, $references, $transformationHelper);
  }

  /**
   * Validates a table node.
   */
  protected function validateTableNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateSimpleContentNode('table', $node, $path, $errors, $references, $transformationHelper);
  }

  /**
   * Validates a table_row node.
   */
  protected function validateTableRowNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateSimpleContentNode('table_row', $node, $path, $errors, $references, $transformationHelper);
  }

  /**
   * Validates a table_cell node.
   */
  protected function validateTableCellNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateSimpleContentNode('table_cell', $node, $path, $errors, $references, $transformationHelper);
  }

  /**
   * Validates a table_header node.
   */
  protected function validateTableHeaderNode(array $node, array $path, array &$errors, array &$references, $transformationHelper): array {
    return $this->validateSimpleContentNode('table_header', $node, $path, $errors, $references, $transformationHelper);
  }

}
