<?php

namespace Drupal\prosemirror\Plugin\ProseMirror\Rendering;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\prosemirror\Rendering\Plugin\ProseMirrorRenderingPluginBase;
use Drupal\prosemirror\Rendering\ProseMirrorRenderer;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Renders code blocks.
 *
 * @ProseMirrorRenderingPlugin(
 *   id = "code_block",
 *   label = @Translation("Code Block Renderer"),
 *   description = @Translation("Renders code blocks."),
 *   node_types = {"code_block"},
 *   formats = {"html"},
 *   weight = 25
 * )
 */
class CodeBlockRenderer extends ProseMirrorRenderingPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a CodeBlockRenderer.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function render(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    if (!$this->supportsFormat($format)) {
      return FALSE;
    }

    $type = $node['type'];

    switch ($type) {
      case 'code_block':
        return $this->renderCodeBlock($node, $renderArray, $format, $renderer);
    }

    return FALSE;
  }

  /**
   * Renders a code block element.
   *
   * @param array $node
   *   The code block node.
   * @param array &$renderArray
   *   The render array to populate.
   * @param string $format
   *   The target format.
   * @param \Drupal\prosemirror\Rendering\ProseMirrorRenderer $renderer
   *   The renderer service.
   *
   * @return bool
   *   TRUE if rendered successfully.
   */
  protected function renderCodeBlock(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    $attrs = $node['attrs'] ?? [];
    $variant = $attrs['variant'] ?? '';
    $name = $attrs['name'] ?? '';

    // Render child content.
    $children = [];
    if (isset($node['content']) && is_array($node['content'])) {
      $children = $this->renderChildren($node['content'], $format, $renderer);
    }

    $classes = ['pm-code-block'];
    if (!empty($variant)) {
      $classes[] = 'pm-code-block-' . $variant;
    }

    $renderArray = [
      '#type' => 'html_tag',
      '#tag' => 'pre',
      '#attributes' => ['class' => $classes],
      'code' => [
        '#type' => 'html_tag',
        '#tag' => 'code',
        '#attributes' => ['class' => ['pm-code-content']],
        'children' => $children,
      ],
    ];

    // Add filename if provided.
    if (!empty($name)) {
      $renderArray['filename'] = [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#attributes' => ['class' => ['pm-code-filename']],
        '#plain_text' => $name,
        '#weight' => -10,
      ];
    }

    return TRUE;
  }

}
