<?php

namespace Drupal\prosemirror\Plugin\ProseMirror\Rendering;

use Drupal\prosemirror\Rendering\Plugin\ProseMirrorRenderingPluginBase;
use Drupal\prosemirror\Rendering\ProseMirrorRenderer;

/**
 * Renders default elements like steps, info boxes, columns, tabs, etc.
 *
 * @todo move to `prosemirror_defaults` submodule.
 *
 * @ProseMirrorRenderingPlugin(
 *   id = "default_elements",
 *   label = @Translation("Default Elements Renderer"),
 *   description = @Translation("Renders default elements like steps, info boxes, columns, tabs, etc. Can be overridden by custom rendering plugins."),
 *   node_types = {"steps", "step", "info_box", "columns", "column"},
 *   formats = {"html"},
 *   weight = 100
 * )
 */
class DefaultElementsRenderer extends ProseMirrorRenderingPluginBase {

  /**
   * {@inheritdoc}
   */
  public function render(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    if (!$this->supportsFormat($format)) {
      return FALSE;
    }

    $type = $node['type'];
    $attrs = $node['attrs'] ?? [];

    switch ($type) {
      case 'steps':
        return $this->renderSteps($node, $renderArray, $format, $renderer);

      case 'step':
        return $this->renderStep($node, $renderArray, $format, $renderer, $attrs);

      case 'info_box':
        return $this->renderInfoBox($node, $renderArray, $format, $renderer, $attrs);

      case 'columns':
        return $this->renderColumns($node, $renderArray, $format, $renderer);

      case 'column':
        return $this->renderColumn($node, $renderArray, $format, $renderer);
    }

    return FALSE;
  }

  /**
   * Renders a steps container.
   */
  protected function renderSteps(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    $children = [];
    if (isset($node['content']) && is_array($node['content'])) {
      $children = $this->renderChildren($node['content'], $format, $renderer);
    }

    $renderArray = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => ['class' => ['pm-steps', 'steps']],
      '#attached' => ['library' => ['prosemirror/html-rendering']],
      'children' => $children,
    ];

    return TRUE;
  }

  /**
   * Renders a single step.
   */
  protected function renderStep(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer, array $attrs): bool {
    $children = [];
    if (isset($node['content']) && is_array($node['content'])) {
      $children = $this->renderChildren($node['content'], $format, $renderer);
    }

    $name = $attrs['name'] ?? '';
    $index = $attrs['index'] ?? NULL;

    $classes = ['pm-step', 'step'];

    $renderArray = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => ['class' => $classes],
      'children' => $children,
    ];

    // Add step number if provided.
    if ($index !== NULL) {
      $renderArray['step_number'] = [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#attributes' => ['class' => ['pm-step-number']],
        '#value' => (string) $index,
        '#weight' => -10,
      ];
    }

    // Add step name if provided.
    if (!empty($name)) {
      $renderArray['step_name'] = [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#attributes' => ['class' => ['pm-step-name']],
        '#plain_text' => $name,
        '#weight' => -5,
      ];
    }

    return TRUE;
  }

  /**
   * Renders an info box.
   */
  protected function renderInfoBox(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer, array $attrs): bool {
    $children = [];
    if (isset($node['content']) && is_array($node['content'])) {
      $children = $this->renderChildren($node['content'], $format, $renderer);
    }

    $variant = $attrs['variant'] ?? 'info';
    $name = $attrs['name'] ?? '';

    $message_type = $variant==='error' || $variant==='warning' ? $variant : 'status';

    $headings = [];

    // Add title if provided.
    if (!empty($name)) {
      $headings[$message_type] = $name;
    }

    $messages = [];
    $messages[$message_type] = [$children];

    $renderArray = [
      '#theme' => 'status_messages',
      '#message_list' => $messages,
    ] + (empty($headings) ? [] : [
      '#status_headings' => $headings,
    ]);

    return TRUE;
  }

  /**
   * Renders a columns container.
   */
  protected function renderColumns(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    $children = [];
    if (isset($node['content']) && is_array($node['content'])) {
      $children = $this->renderChildren($node['content'], $format, $renderer);
    }

    $renderArray = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => ['class' => ['pm-columns', 'columns']],
      '#attached' => ['library' => ['prosemirror/html-rendering']],
      'children' => $children,
    ];

    return TRUE;
  }

  /**
   * Renders a single column.
   */
  protected function renderColumn(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    $children = [];
    if (isset($node['content']) && is_array($node['content'])) {
      $children = $this->renderChildren($node['content'], $format, $renderer);
    }

    $renderArray = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => ['class' => ['pm-column', 'column']],
      'children' => $children,
    ];

    return TRUE;
  }
}
