<?php

namespace Drupal\prosemirror\Plugin\ProseMirror\Rendering;

use Drupal\prosemirror\Rendering\Plugin\ProseMirrorRenderingPluginBase;
use Drupal\prosemirror\Rendering\ProseMirrorRenderer;

/**
 * Renders default elements like tables.
 *
 * @ProseMirrorRenderingPlugin(
 *   id = "system_elements",
 *   label = @Translation("System Elements Renderer"),
 *   description = @Translation("Renders system elements like tables."),
 *   node_types = {"table", "table_row", "table_cell", "table_header"},
 *   formats = {"html"},
 *   weight = 100
 * )
 */
class SystemElementsRenderer extends ProseMirrorRenderingPluginBase {

  /**
   * {@inheritdoc}
   */
  public function render(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    if (!$this->supportsFormat($format)) {
      return FALSE;
    }

    $type = $node['type'];

    switch ($type) {
      case 'table':
        return $this->renderTable($node, $renderArray, $format, $renderer);
    }

    return FALSE;
  }

  /**
   * Renders a steps container.
   */
  protected function renderTable(array $node, array &$renderArray, string $format, ProseMirrorRenderer $renderer): bool {
    $name = $node['attrs']['name'] ?? '';

    $renderArray = [
      '#theme' => 'table',
      '#attributes' => ['class' => ['pm-table']],
    ];
    if($name) {
      $renderArray['#caption'] = $name;
    }

    if (isset($node['content']) && is_array($node['content'])) {
      foreach($node['content'] as $index=>$row) {
        // Skip invalid children.
        if($row['type'] !== 'table_row') {
          continue;
        }

        // Skip empty rows.
        if(!isset($row['content']) || !is_array($row['content']) || !count($row['content'])) {
          continue;
        }

        $is_header_row = $row['content'][0]['type']==='table_header' && (count($row['content'])===1 || $row['content'][1]['type']==='table_header');

        foreach($row['content'] as $cell) {
          // Skip invalid cells.
          if(!in_array($cell['type'], ['table_cell', 'table_header'])) {
            continue;
          }

          $colspan = $cell['attrs']['colspan'] ?? 1;
          $rowspan = $cell['attrs']['rowspan'] ?? 1;

          if($is_header_row) {
            $renderArray['#header'][] = [
              'data' => $this->renderChildren([$cell], $format, $renderer)[0],
            ] + ($colspan > 1 ? ['colspan' => $colspan] : []) + ($rowspan > 1 ? ['rowspan' => $rowspan] : []);
            continue;
          }

          $renderArray['#rows']['row'.$index][] = [
            'data' => $this->renderChildren([$cell], $format, $renderer)[0],
          ] + ($colspan > 1 ? ['colspan' => $colspan] : []) + ($rowspan > 1 ? ['rowspan' => $rowspan] : []) + ($cell['type'] === 'table_header' ? ['header' => TRUE] : []);
        }
      }
    }

    return TRUE;
  }
}
