<?php

namespace Drupal\prosemirror\Plugin\Validation\Constraint;

use Drupal\Core\Field\FieldItemListInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validates the ProseMirrorContentConstraint constraint.
 */
class ProseMirrorContentConstraintValidator extends ConstraintValidator {

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint) {
    if (!$constraint instanceof ProseMirrorContentConstraint) {
      throw new UnexpectedTypeException($constraint, ProseMirrorContentConstraint::class);
    }

    // Skip validation if value is empty.
    if (empty($value)) {
      return;
    }

    // For field items, we need to get the actual value and check the format.
    $format = NULL;
    if ($value instanceof FieldItemListInterface) {
      // Get the format from the first item.
      if ($value->first()) {
        $format = $value->first()->format ?? NULL;
      }
      $value = $value->value;
    }

    // Skip validation if not using ProseMirror format.
    if (!$format || !$this->isProseMirrorFormat($format)) {
      return;
    }

    // Skip if not a string or doesn't look like JSON.
    if (!is_string($value) || !$this->isJson($value)) {
      return;
    }

    try {
      $json_data = json_decode($value, TRUE);
      if (json_last_error() !== JSON_ERROR_NONE) {
        $this->context->addViolation($constraint->invalidContent);
        return;
      }

      // Get the transformation helper service.
      /** @var \Drupal\prosemirror\Transformation\TransformationHelper $transformation_helper */
      $transformation_helper = \Drupal::service('prosemirror.transformation_helper');

      // Validate the ProseMirror content.
      $result = $transformation_helper->validateAndSanitize($json_data);

      if (!$result->isValid()) {
        foreach ($result->getErrors() as $error) {
          $this->context->addViolation($error->getMessage());
        }
      }
    }
    catch (\Exception $e) {
      \Drupal::messenger()->addError('ProseMirror validation error: ' . $e->getMessage());
      $this->context->addViolation($constraint->invalidContent);
    }
  }

  /**
   * Check if a text format uses ProseMirror.
   *
   * @param string $format_id
   *   The format ID to check.
   *
   * @return bool
   *   TRUE if the format uses ProseMirror, FALSE otherwise.
   */
  protected function isProseMirrorFormat($format_id) {
    $editor_storage = \Drupal::entityTypeManager()->getStorage('editor');
    $editor = $editor_storage->load($format_id);

    if ($editor) {
      return $editor->getEditor() === 'prosemirror';
    }

    return FALSE;
  }

  /**
   * Check if a string is valid JSON.
   *
   * @param string $string
   *   The string to check.
   *
   * @return bool
   *   TRUE if the string is valid JSON, FALSE otherwise.
   */
  protected function isJson($string) {
    if (!is_string($string)) {
      return FALSE;
    }
    json_decode($string);
    return json_last_error() === JSON_ERROR_NONE;
  }

}
