<?php

declare(strict_types=1);

namespace Drupal\prosemirror\Transformation;

use Drupal\Component\Uuid\Uuid;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;

/**
 * Represents an entity reference found during content transformation.
 *
 * E.g., a reference to a node using a link mark or a media item.
 */
class EntityReference {

  /**
   * The entity type (e.g., 'node', 'media').
   *
   * @var string
   */
  protected string $entityType;

  /**
   * The entity UUID.
   *
   * @var string
   */
  protected string $entityUuid;

  /**
   * Constructs an EntityReference.
   *
   * @param string $entityType
   *   The entity type.
   * @param string $entityUuid
   *   The entity UUID.
   */
  public function __construct(string $entityType, string $entityUuid) {
    $this->entityType = $entityType;
    $this->entityUuid = $entityUuid;
  }

  /**
   * Gets the entity type.
   *
   * @return string
   *   The entity type.
   */
  public function getEntityType(): string {
    return $this->entityType;
  }

  /**
   * Gets the entity UUID.
   *
   * @return string
   *   The entity UUID.
   */
  public function getEntityUuid(): string {
    return $this->entityUuid;
  }

  /**
   * Converts to array for JSON serialization.
   *
   * @return array
   *   The array representation.
   */
  public function toArray(): array {
    return [
      'entity_type' => $this->entityType,
      'entity_uuid' => $this->entityUuid,
    ];
  }

  /**
   * Checks if the entity reference is valid.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface|null $entity_repository
   *   The entity repository service.
   *
   * @return bool
   *   TRUE if the entity reference is valid, FALSE otherwise.
   */
  public function isValid(?EntityRepositoryInterface $entity_repository = NULL): bool {
    if (!Uuid::isValid($this->entityUuid)) {
      return FALSE;
    }

    if ($entity_repository) {
      if (!$entity_repository->loadEntityByUuid($this->entityType, $this->entityUuid)) {
        return FALSE;
      }
    }

    return TRUE;
  }

}
