<?php

declare(strict_types=1);

namespace Drupal\prosemirror\Transformation;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Drupal\prosemirror\Element\ElementProvider;
use Psr\Log\LoggerInterface;

/**
 * Helper service for validating and sanitizing ProseMirror content.
 */
class TransformationHelper {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected EntityRepositoryInterface $entityRepository;

  /**
   * The ProseMirror element type plugin manager.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager
   */
  protected ProseMirrorElementTypeManager $elementTypeManager;

  /**
   * The element provider.
   *
   * @var \Drupal\prosemirror\Element\ElementProvider
   */
  protected ElementProvider $elementProvider;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Cache of loaded elements.
   *
   * @var array|null
   */
  protected ?array $elementsCache = NULL;

  /**
   * Cache of loaded marks.
   *
   * @var array|null
   */
  protected ?array $marksCache = NULL;

  /**
   * Constructs a TransformationHelper.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager $element_type_manager
   *   The ProseMirror element type plugin manager.
   * @param \Drupal\prosemirror\Element\ElementProvider $element_provider
   *   The element provider.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityRepositoryInterface $entity_repository,
    ProseMirrorElementTypeManager $element_type_manager,
    ElementProvider $element_provider,
    LoggerInterface $logger,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityRepository = $entity_repository;
    $this->elementTypeManager = $element_type_manager;
    $this->elementProvider = $element_provider;
    $this->logger = $logger;
  }

  /**
   * Gets the entity repository.
   *
   * @return \Drupal\Core\Entity\EntityRepositoryInterface
   *   The entity repository.
   */
  public function getEntityRepository(): EntityRepositoryInterface {
    return $this->entityRepository;
  }

  /**
   * Validates and sanitizes ProseMirror content.
   *
   * @param array $content
   *   The content to validate and sanitize.
   *
   * @return \Drupal\prosemirror\Transformation\ElementInstance
   *   The element instance with validation results.
   */
  public function validateAndSanitize(array $content): ElementInstance {
    $errors = [];
    $references = [];

    $sanitized = $this->sanitizeNode($content, [], $errors, $references);

    return new ElementInstance($sanitized, $references, $errors);
  }

  /**
   * Enriches ProseMirror content for display during editor loading.
   *
   * This method adds display-specific attributes that were stripped during
   * validation/serialization, such as entity labels and internal URLs.
   *
   * @param array $content
   *   The content to enrich.
   *
   * @return array
   *   The enriched content with display attributes added.
   */
  public function enrichContentForDisplay(array $content): array {
    return $this->enrichNode($content);
  }

  /**
   * Recursively enriches a node and its children for display.
   *
   * @param array $node
   *   The node to enrich.
   *
   * @return array
   *   The enriched node.
   */
  protected function enrichNode(array $node): array {
    $enriched = $node;

    // Handle attributes enrichment.
    if (isset($node['attrs']) && is_array($node['attrs'])) {
      $enriched['attrs'] = $this->enrichAttributes($node, $node['attrs']);
    }

    // Handle marks enrichment.
    if (isset($node['marks']) && is_array($node['marks'])) {
      $enriched['marks'] = $this->enrichMarks($node['marks']);
    }

    // Handle content enrichment.
    if (isset($node['content']) && is_array($node['content'])) {
      $enriched['content'] = [];
      foreach ($node['content'] as $child) {
        $enriched['content'][] = $this->enrichNode($child);
      }
    }

    return $enriched;
  }

  /**
   * Enriches node attributes for display.
   *
   * @param array $node
   *   The complete node data.
   * @param array $attrs
   *   The attributes to enrich.
   *
   * @return array
   *   The enriched attributes.
   */
  protected function enrichAttributes(array $node, array $attrs): array {
    if (!isset($node['type'])) {
      return $attrs;
    }

    $type = $node['type'];

    // Load the element configuration.
    $element = $this->getElement($type);
    if (!$element) {
      return $attrs;
    }

    // Get element type plugin.
    $elementType = $element->getType();
    $plugin = NULL;

    try {
      $plugin = $this->elementTypeManager->createInstance($elementType, $element->getOptions() ?? []);
    }
    catch (\Exception $e) {
      // Return attributes as-is as fallback.
      return $attrs;
    }

    // If we have a plugin, let it handle the enrichment.
    if ($plugin && method_exists($plugin, 'enrichAttributesForEditorDisplay')) {
      return $plugin->enrichAttributesForEditorDisplay($attrs, $node);
    }

    return $attrs;
  }

  /**
   * Enriches marks for display.
   *
   * @param array $marks
   *   The marks to enrich.
   *
   * @return array
   *   The enriched marks.
   */
  protected function enrichMarks(array $marks): array {
    $enriched = [];

    foreach ($marks as $mark) {
      if (!isset($mark['type'])) {
        $enriched[] = $mark;
        continue;
      }

      $markType = $mark['type'];

      // Check if mark is configured.
      $markEntity = $this->getMark($markType);
      if (!$markEntity) {
        $enriched[] = $mark;
        continue;
      }

      $enrichedMark = [
        'type' => $markType,
      ];

      // Handle mark attributes if present.
      if (isset($mark['attrs']) && is_array($mark['attrs'])) {
        $enrichedMark['attrs'] = $this->enrichMarkAttributes($mark['attrs'], $markType);
      }

      $enriched[] = $enrichedMark;
    }

    return $enriched;
  }

  /**
   * Enriches mark attributes for display.
   *
   * @param array $attrs
   *   The mark attributes to enrich.
   * @param string $markType
   *   The mark type.
   *
   * @return array
   *   The enriched mark attributes.
   */
  protected function enrichMarkAttributes(array $attrs, string $markType): array {
    // Special handling for link marks.
    if ($markType === 'link' && isset($attrs['linkType']) && $attrs['linkType'] === 'internal') {
      return $this->enrichLinkMarkAttributes($attrs);
    }

    return $attrs;
  }

  /**
   * Enriches link mark attributes for display.
   *
   * @param array $attrs
   *   The link mark attributes to enrich.
   *
   * @return array
   *   The enriched link mark attributes.
   */
  protected function enrichLinkMarkAttributes(array $attrs): array {
    $enriched = $attrs;

    // Only enrich if we have entity reference information.
    if (!isset($attrs['entityUuid']) || !isset($attrs['entityType'])) {
      return $enriched;
    }

    $entityUuid = $attrs['entityUuid'];
    $entityType = $attrs['entityType'];

    try {
      // Load the entity.
      $entity = $this->entityRepository->loadEntityByUuid($entityType, $entityUuid);

      if ($entity) {
        // Add entity label.
        if (method_exists($entity, 'label')) {
          $enriched['entityLabel'] = $entity->label();
        }

        // Add entity URL.
        if (method_exists($entity, 'toUrl')) {
          try {
            $url = $entity->toUrl();
            $enriched['entityUrl'] = $url->toString();
          }
          catch (\Exception $e) {
            // If URL generation fails, don't add the attribute.
            $this->logger->warning('Could not generate URL for entity @type/@uuid: @message', [
              '@type' => $entityType,
              '@uuid' => $entityUuid,
              '@message' => $e->getMessage(),
            ]);
          }
        }
      }
    }
    catch (\Exception $e) {
      // If entity loading fails, log the error but don't break the enrichment.
      $this->logger->warning('Could not load entity @type/@uuid for link enrichment: @message', [
        '@type' => $entityType,
        '@uuid' => $entityUuid,
        '@message' => $e->getMessage(),
      ]);
    }

    return $enriched;
  }

  /**
   * Sanitizes a ProseMirror node.
   *
   * @param array $node
   *   The node to sanitize.
   * @param array $path
   *   The current path in the document tree.
   * @param array &$errors
   *   Array to collect validation errors.
   * @param array &$references
   *   Array to collect entity references.
   *
   * @return array
   *   The sanitized node.
   */
  protected function sanitizeNode(array $node, array $path, array &$errors, array &$references): array {
    // Required properties.
    if (!isset($node['type'])) {
      $errors[] = ValidationError::atPath('Node missing required "type" property', $path);
      return [];
    }

    $type = $node['type'];

    // Load the element configuration.
    $element = $this->getElement($type);
    if (!$element) {
      $errors[] = ValidationError::atPath(sprintf('Unknown node type: %s', $type), $path);
      return [];
    }

    // Get element type plugin.
    $elementType = $element->getType();
    $plugin = NULL;

    try {
      $plugin = $this->elementTypeManager->createInstance($elementType, $element->getOptions() ?? []);
    }
    catch (\Exception $e) {
      // System elements might not have plugins - use default validation.
    }

    // If we have a plugin, let it handle the validation.
    if ($plugin && method_exists($plugin, 'validateNode')) {
      return $plugin->validateNode($node, $path, $errors, $references, $this);
    }

    // Default validation for system elements without plugins.
    $sanitized = [
      'type' => $type,
    ];

    // Handle attrs if present.
    if (isset($node['attrs']) && is_array($node['attrs'])) {
      $sanitized['attrs'] = $this->sanitizeAttrs($element, $node['attrs'], $path, $errors, $references);
    }

    // Handle marks if present.
    if (isset($node['marks']) && is_array($node['marks'])) {
      $sanitized['marks'] = $this->sanitizeMarks($node['marks'], $path, $errors, $references);
    }

    // Handle content if present.
    if (isset($node['content']) && is_array($node['content'])) {
      $sanitized['content'] = [];
      foreach ($node['content'] as $index => $child) {
        $childPath = array_merge($path, ['content', $index]);
        $sanitizedChild = $this->sanitizeNode($child, $childPath, $errors, $references);
        if (!empty($sanitizedChild)) {
          $sanitized['content'][] = $sanitizedChild;
        }
      }
    }

    if ($element->getContentMin() && (!isset($sanitized['content']) || count($sanitized['content']) < $element->getContentMin())) {
      $errors[] = ValidationError::atPath('Content count is less than the minimum allowed of ' . $element->getContentMin(), $path);
    }

    if ($element->getContentMax() && isset($sanitized['content']) && count($sanitized['content']) > $element->getContentMax()) {
      $errors[] = ValidationError::atPath('Content count is greater than the maximum allowed of ' . $element->getContentMax(), $path);
    }

    // Handle text nodes.
    if (isset($node['text'])) {
      $sanitized['text'] = (string) $node['text'];
    }

    return $sanitized;
  }

  /**
   * Validates a child node (public method for plugins to use).
   *
   * @param array $node
   *   The node to validate.
   * @param array $path
   *   The current path in the document tree.
   * @param array &$errors
   *   Array to collect validation errors.
   * @param array &$references
   *   Array to collect entity references.
   *
   * @return array
   *   The sanitized node.
   */
  public function validateChildNode(array $node, array $path, array &$errors, array &$references): array {
    return $this->sanitizeNode($node, $path, $errors, $references);
  }

  /**
   * Sanitizes node attributes based on element configuration.
   *
   * This is now primarily for system elements without plugins.
   *
   * @param \Drupal\prosemirror\ProseMirrorElementInterface $element
   *   The element configuration.
   * @param array $attrs
   *   The attributes to sanitize.
   * @param array $path
   *   The current path.
   * @param array &$errors
   *   Array to collect errors.
   * @param array &$references
   *   Array to collect references.
   *
   * @return array
   *   The sanitized attributes.
   */
  protected function sanitizeAttrs($element, array $attrs, array $path, array &$errors, array &$references): array {
    $sanitized = [];

    // Handle special system types that don't have plugins.
    switch ($element->id()) {
      case 'text':
        // Text nodes don't have attributes.
        break;

      case 'media':
        // Media requires entity type and UUID.
        $entityType = $attrs['data-entity-type'] ?? $attrs['entity-type'] ?? NULL;
        $entityUuid = $attrs['data-entity-uuid'] ?? $attrs['entity-uuid'] ?? NULL;

        if ($entityType === 'media' && $entityUuid) {
          $sanitized['data-entity-type'] = 'media';
          $sanitized['data-entity-uuid'] = (string) $entityUuid;

          $reference = new EntityReference('media', $entityUuid);

          // Add reference.
          if ($reference->isValid($this->entityRepository)) {
            $references[] = $reference;
          }
          else {
            $errors[] = ValidationError::atPath('Invalid media reference', $path);
          }
        }
        else {
          $errors[] = ValidationError::atPath('Media node missing required entity-type or entity-uuid', $path);
        }
        break;

      case 'table_cell':
      case 'table_header':
        if (isset($attrs['colspan'])) {
          $colspan = filter_var($attrs['colspan'], FILTER_VALIDATE_INT);
          if ($colspan !== FALSE && $colspan >= 1) {
            $sanitized['colspan'] = $colspan;
          }
        }
        if (isset($attrs['rowspan'])) {
          $rowspan = filter_var($attrs['rowspan'], FILTER_VALIDATE_INT);
          if ($rowspan !== FALSE && $rowspan >= 1) {
            $sanitized['rowspan'] = $rowspan;
          }
        }
        if (isset($attrs['colwidth'])) {
          if (is_array($attrs['colwidth'])) {
            $sanitized['colwidth'] = array_map('intval', $attrs['colwidth']);
          }
          else {
            $colwidth = filter_var($attrs['colwidth'], FILTER_VALIDATE_INT);
            if ($colwidth !== FALSE) {
              $sanitized['colwidth'] = $colwidth;
            }
          }
        }
        break;
    }

    return $sanitized;
  }

  /**
   * Sanitizes marks array.
   *
   * @param array $marks
   *   The marks to sanitize.
   * @param array $path
   *   The current path.
   * @param array &$errors
   *   Array to collect errors.
   * @param array &$references
   *   Array to collect entity references.
   *
   * @return array
   *   The sanitized marks.
   */
  public function sanitizeMarks(array $marks, array $path, array &$errors, array &$references = []): array {
    $sanitized = [];

    foreach ($marks as $index => $mark) {
      if (!isset($mark['type'])) {
        $errors[] = ValidationError::atPath('Mark missing required "type" property', array_merge($path, [
          'marks',
          $index,
        ]));
        continue;
      }

      $markType = $mark['type'];

      // Check if mark is configured.
      $markEntity = $this->getMark($markType);
      if (!$markEntity) {
        $errors[] = ValidationError::atPath(sprintf('Unknown mark type: %s', $markType), array_merge($path, [
          'marks',
          $index,
        ]));
        continue;
      }

      $sanitizedMark = [
        'type' => $markType,
      ];

      // Handle mark attributes if present.
      if (isset($mark['attrs']) && is_array($mark['attrs'])) {
        $sanitizedAttrs = [];
        $markAttrs = $markEntity->getAttributes();

        // Special handling for link marks.
        if ($markType === 'link') {
          if (!isset($mark['attrs']['linkType'])) {
            $errors[] = ValidationError::atPath('Link mark missing required linkType attribute', array_merge($path, [
              'marks',
              $index,
            ]));
            continue;
          }

          // Validate href.
          if (!isset($mark['attrs']['href']) || empty($mark['attrs']['href'])) {
            $errors[] = ValidationError::atPath('Link mark missing required href attribute', array_merge($path, [
              'marks',
              $index,
            ]));
            continue;
          }

          $href = $mark['attrs']['href'];

          if (isset($mark['attrs']['variant'])) {
            $sanitizedAttrs['variant'] = (string) $mark['attrs']['variant'];
          }

          if ($mark['attrs']['linkType'] === 'internal') {
            // Entity reference attributes.
            if (isset($mark['attrs']['entityUuid']) && isset($mark['attrs']['entityType'])) {
              $sanitizedAttrs['entityUuid'] = (string) $mark['attrs']['entityUuid'];
              $sanitizedAttrs['entityType'] = (string) $mark['attrs']['entityType'];

              if (!in_array($sanitizedAttrs['entityType'], ['node', 'media', 'block_content'])) {
                $errors[] = ValidationError::atPath('Invalid entity type for link mark', array_merge($path, [
                  'marks',
                  $index,
                ]));
                continue;
              }

              $reference = new EntityReference($sanitizedAttrs['entityType'], $sanitizedAttrs['entityUuid']);

              // Add reference.
              if ($reference->isValid($this->entityRepository)) {
                $references[] = $reference;
              }
              else {
                $errors[] = ValidationError::atPath('Invalid content reference', array_merge($path, [
                  'marks',
                  $index,
                ]));
                continue;
              }

              $sanitizedAttrs['href'] = 'entity:' . $sanitizedAttrs['entityType'] . '/' . $sanitizedAttrs['entityUuid'];
            }
            else {
              $errors[] = ValidationError::atPath('Link mark missing required entityUuid and entityType attributes', array_merge($path, [
                'marks',
                $index,
              ]));
              continue;
            }

            $sanitizedAttrs['linkType'] = 'internal';
          }
          elseif ($mark['attrs']['linkType'] === 'external') {
            // Basic URL validation.
            if (!filter_var($href, FILTER_VALIDATE_URL)) {
              $errors[] = ValidationError::atPath('Invalid link href format', array_merge($path, ['marks', $index]));
              continue;
            }

            $sanitizedAttrs['href'] = $href;
            $sanitizedAttrs['linkUri'] = $href;
            $sanitizedAttrs['linkType'] = 'external';
          }
        }

        if (!empty($sanitizedAttrs)) {
          $sanitizedMark['attrs'] = $sanitizedAttrs;
        }
      }

      $sanitized[] = $sanitizedMark;
    }

    return $sanitized;
  }

  /**
   * Gets a ProseMirror element by type.
   *
   * @param string $type
   *   The element type.
   *
   * @return \Drupal\prosemirror\ProseMirrorElementInterface|null
   *   The element or NULL if not found.
   */
  protected function getElement(string $type): ?object {
    if ($this->elementsCache === NULL) {
      $this->loadElements();
    }

    return $this->elementsCache[$type] ?? NULL;
  }

  /**
   * Gets a ProseMirror mark by type.
   *
   * @param string $type
   *   The mark type.
   *
   * @return \Drupal\prosemirror\ProseMirrorMarkInterface|null
   *   The mark or NULL if not found.
   */
  protected function getMark(string $type): ?object {
    if ($this->marksCache === NULL) {
      $this->loadMarks();
    }

    return $this->marksCache[$type] ?? NULL;
  }

  /**
   * Loads all configured elements.
   */
  protected function loadElements(): void {
    $this->elementsCache = [];

    try {
      // Load from element provider which includes both configured and system elements.
      $elements = $this->elementProvider->getAllElements();

      foreach ($elements as $element) {
        $this->elementsCache[$element->id()] = $element;
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to load ProseMirror elements: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Loads all configured marks.
   */
  protected function loadMarks(): void {
    $this->marksCache = [];

    try {
      // Load from element provider which now handles marks.
      $marks = $this->elementProvider->getAllMarks();

      foreach ($marks as $mark) {
        $this->marksCache[$mark->id()] = $mark;
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to load ProseMirror marks: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

}
