<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Plugin\ProseMirror\ElementType;

use Psr\Log\LoggerInterface;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\prosemirror\Entity\ProseMirrorElement;
use Drupal\prosemirror\Element\ElementProvider;
use Drupal\prosemirror\Plugin\ProseMirror\ElementType\BaseNode;
use Drupal\prosemirror\Transformation\TransformationHelper;
use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\Tests\prosemirror\Unit\ProseMirrorElementTestTrait;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;
use Prophecy\Argument;

/**
 * Tests the BaseNode plugin.
 *
 * @group prosemirror
 * @coversDefaultClass \Drupal\prosemirror\Plugin\ProseMirror\ElementType\BaseNode
 */
class BaseNodeTest extends UnitTestCase {

  use ProphecyTrait;
  use ProseMirrorElementTestTrait;

  /**
   * The base node plugin.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirror\ElementType\BaseNode
   */
  protected BaseNode $baseNode;

  /**
   * Mock transformation helper.
   *
   * @var \Drupal\prosemirror\Transformation\TransformationHelper|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $transformationHelper;

  /**
   * Mock element provider.
   *
   * @var \Drupal\prosemirror\Element\ElementProvider|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementProvider;

  /**
   * Mock entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    [, $this->entityRepository] = $this->createMockEntityServices();
    $this->elementProvider = $this->prophesize(ElementProvider::class);

    // Setup elements and marks using trait.
    $this->setupElementProviderMock($this->elementProvider);

    // Create service container with element provider and entity repository.
    $container = new ContainerBuilder();
    $container->set('prosemirror.element_provider', $this->elementProvider->reveal());
    $container->set('entity.repository', $this->entityRepository->reveal());
    \Drupal::setContainer($container);

    // Create base node plugin.
    $this->baseNode = new BaseNode([], 'base_node', []);

    // Mock transformation helper.
    $this->transformationHelper = $this->prophesize(TransformationHelper::class);
  }

  /**
   * Tests heading validation with correct level.
   *
   * @covers ::validateAttributes
   */
  public function testHeadingWithCorrectLevel(): void {
    $attrs = ['level' => 2];
    $errors = [];
    $references = [];

    $result = $this->baseNode->validateAttributes($attrs, [], $errors, $references);

    // BaseNode validateAttributes now returns empty array.
    $this->assertEquals([], $result);
    $this->assertEmpty($errors);
  }

  /**
   * Tests heading validation without level attribute.
   *
   * @covers ::validateNode
   */
  public function testHeadingWithoutLevel(): void {
    $node = [
      'type' => 'heading_3',
      'content' => [
        ['type' => 'text', 'text' => 'Title'],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Create actual transformation helper with mocked dependencies.
    $entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $logger = $this->prophesize(LoggerInterface::class);

    $transformationHelper = new TransformationHelper(
      $entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $logger->reveal()
    );

    $result = $this->baseNode->validateNode($node, $path, $errors, $references, $transformationHelper);

    // BaseNode validateAttributes returns empty array, so no attrs should be present.
    $this->assertEquals('heading_3', $result['type']);
    $this->assertArrayNotHasKey('attrs', $result);
    $this->assertEmpty($errors);
  }

  /**
   * Tests non-heading node validation.
   *
   * @covers ::validateNode
   */
  public function testNonHeadingNode(): void {
    $node = [
      'type' => 'paragraph',
      'attrs' => ['custom' => 'value', 'class' => 'test-class'],
      'content' => [
        ['type' => 'text', 'text' => 'Some text'],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Create actual transformation helper with mocked dependencies.
    $entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $logger = $this->prophesize(LoggerInterface::class);

    $transformationHelper = new TransformationHelper(
      $entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $logger->reveal()
    );

    $result = $this->baseNode->validateNode($node, $path, $errors, $references, $transformationHelper);

    // BaseNode validateAttributes returns empty array, so no attrs should be present.
    $this->assertEquals('paragraph', $result['type']);
    $this->assertArrayNotHasKey('attrs', $result);
    $this->assertEmpty($errors);
  }

  /**
   * Tests complete node validation with content.
   *
   * @covers ::validateNode
   */
  public function testCompleteNodeValidation(): void {
    $node = [
      'type' => 'heading_1',
      'attrs' => ['level' => 1],
      'content' => [
        ['type' => 'text', 'text' => 'Title'],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Create actual transformation helper with mocked dependencies.
    $entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $logger = $this->prophesize(LoggerInterface::class);

    $transformationHelper = new TransformationHelper(
      $entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $logger->reveal()
    );

    $result = $this->baseNode->validateNode($node, $path, $errors, $references, $transformationHelper);

    // BaseNode validateAttributes returns empty array, so no attrs should be present.
    $this->assertEquals('heading_1', $result['type']);
    $this->assertArrayNotHasKey('attrs', $result);
    $this->assertCount(1, $result['content']);
    $this->assertEmpty($errors);
  }

  /**
   * Tests node validation with marks.
   *
   * @covers ::validateNode
   */
  public function testNodeWithMarks(): void {
    $node = [
      'type' => 'paragraph',
      'attrs' => ['custom' => 'value', 'class' => 'test-class'],
      'marks' => [
        ['type' => 'bold'],
      ],
      'content' => [
        ['type' => 'text', 'text' => 'Some text'],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Create actual transformation helper with mocked dependencies.
    $entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $logger = $this->prophesize(LoggerInterface::class);

    $transformationHelper = new TransformationHelper(
      $entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $logger->reveal()
    );

    $result = $this->baseNode->validateNode($node, $path, $errors, $references, $transformationHelper);

    // BaseNode validateAttributes returns empty array, so no attrs should be present.
    $this->assertEquals('paragraph', $result['type']);
    $this->assertArrayNotHasKey('attrs', $result);
    $this->assertEquals([['type' => 'bold']], $result['marks']);
    $this->assertCount(1, $result['content']);
    $this->assertEmpty($errors);
  }

}
