<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Plugin\ProseMirror\ElementType;

use Psr\Log\LoggerInterface;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\prosemirror\Entity\ProseMirrorElement;
use Drupal\prosemirror\Element\ElementProvider;
use Drupal\prosemirror\Plugin\ProseMirror\ElementType\LeafBlock;
use Drupal\prosemirror\Transformation\TransformationHelper;
use Drupal\Tests\prosemirror\Unit\ProseMirrorElementTestTrait;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;
use Prophecy\Argument;

/**
 * Tests the LeafBlock plugin.
 *
 * @group prosemirror
 * @coversDefaultClass \Drupal\prosemirror\Plugin\ProseMirror\ElementType\LeafBlock
 */
class LeafBlockTest extends UnitTestCase {

  use ProphecyTrait;
  use ProseMirrorElementTestTrait;

  /**
   * The leaf block plugin.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirror\ElementType\LeafBlock
   */
  protected LeafBlock $leafBlock;

  /**
   * Mock transformation helper.
   *
   * @var \Drupal\prosemirror\Transformation\TransformationHelper|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $transformationHelper;

  /**
   * Mock element provider.
   *
   * @var \Drupal\prosemirror\Element\ElementProvider|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementProvider;

  /**
   * Mock entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    [, $this->entityRepository] = $this->createMockEntityServices();
    $this->elementProvider = $this->prophesize(ElementProvider::class);

    // Setup elements and marks using trait.
    $this->setupElementProviderMock($this->elementProvider);

    // Create service container with element provider and entity repository.
    $container = new ContainerBuilder();
    $container->set('prosemirror.element_provider', $this->elementProvider->reveal());
    $container->set('entity.repository', $this->entityRepository->reveal());
    \Drupal::setContainer($container);

    // Create leaf block plugin with test configuration.
    $configuration = [
      'variants' => [
        'primary' => 'Primary',
        'secondary' => 'Secondary',
        'default' => 'Default',
        '456' => 'Test Variant',
        'highlighted' => 'Highlighted',
      ],
    ];
    $this->leafBlock = new LeafBlock($configuration, 'leaf_block', []);

    // Mock transformation helper.
    $this->transformationHelper = $this->prophesize(TransformationHelper::class);
  }

  /**
   * Tests validation with all valid attributes.
   *
   * @covers ::validateAttributes
   */
  public function testValidAttributesAll(): void {
    $attrs = [
      'name' => 'custom-block',
      'variant' => 'primary',
      'class' => 'custom-class',
    ];
    $errors = [];
    $references = [];

    $result = $this->leafBlock->validateAttributes($attrs, [], $errors, $references);

    $this->assertEquals([
      'name' => 'custom-block',
      'variant' => 'primary',
    ], $result);
    $this->assertEmpty($errors);
  }

  /**
   * Tests validation with name only.
   *
   * @covers ::validateAttributes
   */
  public function testValidAttributesNameOnly(): void {
    $attrs = [
      'name' => 'my-block',
    ];
    $errors = [];
    $references = [];

    $result = $this->leafBlock->validateAttributes($attrs, [], $errors, $references);

    $this->assertEquals(['name' => 'my-block'], $result);
    $this->assertEmpty($errors);
  }

  /**
   * Tests validation with variant only.
   *
   * @covers ::validateAttributes
   */
  public function testValidAttributesVariantOnly(): void {
    $attrs = [
      'variant' => 'secondary',
    ];
    $errors = [];
    $references = [];

    $result = $this->leafBlock->validateAttributes($attrs, [], $errors, $references);

    $this->assertEquals(['variant' => 'secondary'], $result);
    $this->assertEmpty($errors);
  }

  /**
   * Tests validation with numeric values (should be cast to string).
   *
   * @covers ::validateAttributes
   */
  public function testAttributesCasting(): void {
    $attrs = [
      'name' => 123,
      'variant' => 456,
    ];
    $errors = [];
    $references = [];

    $result = $this->leafBlock->validateAttributes($attrs, [], $errors, $references);

    $this->assertEquals([
      'name' => '123',
      'variant' => '456',
    ], $result);
    $this->assertEmpty($errors);
  }

  /**
   * Tests validation with runtime attributes (should be stripped).
   *
   * @covers ::validateAttributes
   */
  public function testRuntimeAttributesStripped(): void {
    $attrs = [
      'name' => 'block',
      'variant' => 'default',
      'class' => 'should-be-ignored',
      'updatedAt' => '2024-01-01',
      'editDialog' => TRUE,
      'index' => 1,
    ];
    $errors = [];
    $references = [];

    $result = $this->leafBlock->validateAttributes($attrs, [], $errors, $references);

    $this->assertEquals([
      'name' => 'block',
      'variant' => 'default',
    ], $result);
    $this->assertArrayNotHasKey('class', $result);
    $this->assertArrayNotHasKey('updatedAt', $result);
    $this->assertArrayNotHasKey('editDialog', $result);
    $this->assertArrayNotHasKey('index', $result);
    $this->assertEmpty($errors);
  }

  /**
   * Tests validation with empty attributes.
   *
   * @covers ::validateAttributes
   */
  public function testEmptyAttributes(): void {
    $attrs = [];
    $errors = [];
    $references = [];

    $result = $this->leafBlock->validateAttributes($attrs, [], $errors, $references);

    $this->assertEquals([], $result);
    $this->assertEmpty($errors);
  }

  /**
   * Tests complete node validation.
   *
   * @covers ::validateNode
   */
  public function testCompleteNodeValidation(): void {
    $node = [
      'type' => 'paragraph',
      'attrs' => [
        'name' => 'feature-block',
        'variant' => 'highlighted',
      ],
      'content' => [
        ['type' => 'text', 'text' => 'Content'],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Create actual transformation helper with mocked dependencies.
    $entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $logger = $this->prophesize(LoggerInterface::class);

    $transformationHelper = new TransformationHelper(
      $entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $logger->reveal()
    );

    $result = $this->leafBlock->validateNode($node, $path, $errors, $references, $transformationHelper);

    $this->assertEquals('paragraph', $result['type']);
    $this->assertEquals([
      'name' => 'feature-block',
      'variant' => 'highlighted',
    ], $result['attrs']);
    $this->assertCount(1, $result['content']);
    $this->assertEmpty($errors);
  }

  /**
   * Tests node validation without attributes.
   *
   * @covers ::validateNode
   */
  public function testNodeWithoutAttributes(): void {
    $node = [
      'type' => 'paragraph',
      'content' => [
        ['type' => 'text', 'text' => 'Content'],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Create actual transformation helper with mocked dependencies.
    $entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $logger = $this->prophesize(LoggerInterface::class);

    $transformationHelper = new TransformationHelper(
      $entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $logger->reveal()
    );

    $result = $this->leafBlock->validateNode($node, $path, $errors, $references, $transformationHelper);

    $this->assertEquals('paragraph', $result['type']);
    $this->assertArrayNotHasKey('attrs', $result);
    $this->assertCount(1, $result['content']);
    $this->assertEmpty($errors);
  }

  /**
   * Tests node validation with marks.
   *
   * @covers ::validateNode
   */
  public function testNodeWithMarks(): void {
    $node = [
      'type' => 'paragraph',
      'attrs' => ['name' => 'alert'],
      'marks' => [
        ['type' => 'bold'],
      ],
      'content' => [
        ['type' => 'text', 'text' => 'Alert text'],
      ],
    ];

    $errors = [];
    $references = [];
    $path = [];

    // Create actual transformation helper with mocked dependencies.
    $entityTypeManager = $this->prophesize(EntityTypeManagerInterface::class);
    $elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $logger = $this->prophesize(LoggerInterface::class);

    $transformationHelper = new TransformationHelper(
      $entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $logger->reveal()
    );

    $result = $this->leafBlock->validateNode($node, $path, $errors, $references, $transformationHelper);

    $this->assertEquals('paragraph', $result['type']);
    $this->assertEquals(['name' => 'alert'], $result['attrs']);
    $this->assertEquals([['type' => 'bold']], $result['marks']);
    $this->assertCount(1, $result['content']);
    $this->assertEmpty($errors);
  }

}
