<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Transformation;

use Drupal\Component\Uuid\Uuid;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\prosemirror\Element\ElementProvider;
use Drupal\prosemirror\Entity\ProseMirrorElement;
use Drupal\prosemirror\Entity\ProseMirrorMark;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeInterface;
use Drupal\prosemirror\Transformation\TransformationHelper;
use Drupal\prosemirror\Transformation\ElementInstance;
use Drupal\prosemirror\Transformation\EntityReference;
use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\Tests\prosemirror\Unit\ProseMirrorElementTestTrait;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;
use Prophecy\Argument;
use Psr\Log\LoggerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;

/**
 * Tests attribute whitelisting for all node and mark types.
 *
 * @group prosemirror
 */
class AttributeWhitelistingTest extends UnitTestCase {

  use ProphecyTrait;
  use ProseMirrorElementTestTrait;

  /**
   * The transformation helper service.
   *
   * @var \Drupal\prosemirror\Transformation\TransformationHelper
   */
  protected TransformationHelper $transformationHelper;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityTypeManager;

  /**
   * Mock element type manager.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementTypeManager;

  /**
   * Mock logger.
   *
   * @var \Psr\Log\LoggerInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $logger;

  /**
   * Mock element provider.
   *
   * @var \Drupal\prosemirror\Element\ElementProvider|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementProvider;

  /**
   * Mock entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    [$this->entityTypeManager, $this->entityRepository] = $this->createMockEntityServices();
    $this->elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $this->logger = $this->prophesize(LoggerInterface::class);
    $this->elementProvider = $this->prophesize(ElementProvider::class);

    // Setup elements and marks using trait.
    $this->setupElementProviderMock($this->elementProvider);

    // Create service container with element provider and entity repository.
    $container = new ContainerBuilder();
    $container->set('prosemirror.element_provider', $this->elementProvider->reveal());
    $container->set('entity.repository', $this->entityRepository->reveal());
    \Drupal::setContainer($container);

    // Mock element type plugins.
    $this->setupElementTypePlugins();

    // Create transformation helper with mocked dependencies.
    $this->transformationHelper = new TransformationHelper(
      $this->entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $this->elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $this->logger->reveal()
    );
  }

  /**
   * Sets up element type plugin mocks.
   */
  protected function setupElementTypePlugins(): void {
    // Mock system element plugin for doc and text.
    $systemPlugin = $this->prophesize(ProseMirrorElementTypeInterface::class);
    $systemPlugin->validateNode(
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array'),
      Argument::any()
    )->will(function ($args) {
      $node = $args[0];
      $path = $args[1];
      $errors = &$args[2];
      $references = &$args[3];
      $helper = $args[4];

      if ($node['type'] === 'doc' || $node['type'] === 'text') {
        // System element plugin handles doc and text.
        $result = ['type' => $node['type']];

        // Validate content for doc.
        if ($node['type'] === 'doc' && isset($node['content'])) {
          $result['content'] = [];
          foreach ($node['content'] as $index => $child) {
            $childPath = array_merge($path, ['content', $index]);
            $childResult = $helper->validateChildNode($child, $childPath, $errors, $references);
            if (!empty($childResult)) {
              $result['content'][] = $childResult;
            }
          }
        }

        // Handle text value and marks for text nodes.
        if ($node['type'] === 'text') {
          if (isset($node['text'])) {
            $result['text'] = (string) $node['text'];
          }
          if (isset($node['marks'])) {
            $result['marks'] = $helper->sanitizeMarks($node['marks'], $path, $errors, $references);
          }
        }

        return $result;
      }

      $errors[] = ValidationError::atPath("System element plugin received non-system type: {$node['type']}", $path);
      return [];
    });

    $this->elementTypeManager->createInstance('system', Argument::any())
      ->willReturn($systemPlugin->reveal());

    // Mock base_node plugin for configured elements.
    $baseNodePlugin = $this->prophesize(ProseMirrorElementTypeInterface::class);
    $baseNodePlugin->validateNode(
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array'),
      Argument::any()
    )->will(function ($args) {
      $node = $args[0];
      $path = $args[1];
      $errors = &$args[2];
      $references = &$args[3];
      $helper = $args[4];

      $result = ['type' => $node['type']];

      // Handle different node types.
      if (str_starts_with($node['type'], 'heading')) {
        // Headings keep level attribute.
        if (isset($node['attrs']['level'])) {
          $result['attrs'] = ['level' => (int) $node['attrs']['level']];
        }
      }
      elseif ($node['type'] === 'media') {
        // Media nodes validate entity references.
        $attrs = $node['attrs'] ?? [];
        if (!isset($attrs['data-entity-type']) || !isset($attrs['data-entity-uuid'])) {
          $errors[] = ValidationError::atPath('Media node missing required entity-type or entity-uuid', $path);
          return [];
        }
        if (!Uuid::isValid($attrs['data-entity-uuid'])) {
          $errors[] = ValidationError::atPath('Invalid entity UUID for media node', $path);
          return [];
        }
        $references[] = new EntityReference('media', $attrs['data-entity-uuid']);
        $result['attrs'] = [
          'data-entity-type' => 'media',
          'data-entity-uuid' => $attrs['data-entity-uuid'],
        ];
      }
      elseif ($node['type'] === 'icon') {
        // Icon nodes need iconName.
        $attrs = $node['attrs'] ?? [];
        if (!isset($attrs['iconName'])) {
          $errors[] = ValidationError::atPath('Icon node missing required iconName attribute', $path);
          return [];
        }
        $result['attrs'] = ['iconName' => $attrs['iconName']];
      }
      elseif (in_array($node['type'], ['table_cell', 'table_header'])) {
        // Table cells keep colspan, rowspan, colwidth.
        $attrs = $node['attrs'] ?? [];
        $keepAttrs = [];
        foreach (['colspan', 'rowspan', 'colwidth'] as $attr) {
          if (isset($attrs[$attr])) {
            $keepAttrs[$attr] = $attrs[$attr];
          }
        }
        if (!empty($keepAttrs)) {
          $result['attrs'] = $keepAttrs;
        }
      }
      // Other nodes (paragraph, blockquote, hr, lists, etc.) don't keep attrs.

      // Validate content.
      if (isset($node['content'])) {
        $result['content'] = [];
        foreach ($node['content'] as $index => $child) {
          $childPath = array_merge($path, ['content', $index]);
          $childResult = $helper->validateChildNode($child, $childPath, $errors, $references);
          if (!empty($childResult)) {
            $result['content'][] = $childResult;
          }
        }
      }

      return $result;
    });

    $this->elementTypeManager->createInstance('base_node', Argument::any())
      ->willReturn($baseNodePlugin->reveal());

    // Mock leaf_block plugin for custom elements.
    $leafBlockPlugin = $this->prophesize(ProseMirrorElementTypeInterface::class);
    $leafBlockPlugin->validateNode(
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array'),
      Argument::type('array'),
      Argument::any()
    )->will(function ($args) {
      $node = $args[0];
      $path = $args[1];
      $errors = &$args[2];
      $references = &$args[3];
      $helper = $args[4];

      $result = ['type' => $node['type']];

      // Leaf blocks can have name, variant, class attributes.
      $attrs = $node['attrs'] ?? [];
      $keepAttrs = [];
      foreach (['name', 'variant', 'class'] as $attr) {
        if (isset($attrs[$attr])) {
          $keepAttrs[$attr] = $attrs[$attr];
        }
      }
      if (!empty($keepAttrs)) {
        $result['attrs'] = $keepAttrs;
      }

      // Validate content.
      if (isset($node['content'])) {
        $result['content'] = [];
        foreach ($node['content'] as $index => $child) {
          $childPath = array_merge($path, ['content', $index]);
          $childResult = $helper->validateChildNode($child, $childPath, $errors, $references);
          if (!empty($childResult)) {
            $result['content'][] = $childResult;
          }
        }
      }

      return $result;
    });

    $this->elementTypeManager->createInstance('leaf_block', Argument::any())
      ->willReturn($leafBlockPlugin->reveal());

    // List block is similar.
    $this->elementTypeManager->createInstance('list_block', Argument::any())
      ->willReturn($leafBlockPlugin->reveal());
  }

  /**
   * Tests doc node attribute whitelisting.
   */
  public function testDocNodeAttributes(): void {
    $doc = [
      'type' => 'doc',
      'attrs' => [
        'custom' => 'value',
        'updatedAt' => '2024-01-01',
        'editDialog' => TRUE,
        'index' => 1,
        'meta' => ['key' => 'value'],
      ],
      'content' => [
        [
          'type' => 'paragraph',
          'content' => [
            ['type' => 'text', 'text' => 'Hello'],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($doc);
    $this->assertTrue($result->isValid());

    // Doc nodes should not have any attrs after sanitization.
    $this->assertArrayNotHasKey('attrs', $result->getData());
  }

  /**
   * Tests paragraph node attribute whitelisting.
   */
  public function testParagraphNodeAttributes(): void {
    $paragraph = [
      'type' => 'paragraph',
      'attrs' => [
        'align' => 'center',
        'custom' => 'value',
        'class' => 'paragraph-class',
        'id' => 'para-1',
        'updatedAt' => '2024-01-01',
        'editDialog' => TRUE,
      ],
      'content' => [
        ['type' => 'text', 'text' => 'Hello'],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertTrue($result->isValid());

    // Paragraph nodes should not have attrs after sanitization.
    $this->assertArrayNotHasKey('attrs', $result->getData());
  }

  /**
   * Tests heading node attribute whitelisting.
   */
  public function testHeadingNodeAttributes(): void {
    $heading = [
      'type' => 'heading',
      'attrs' => [
        'level' => 2,
        'align' => 'center',
        'custom' => 'value',
        'class' => 'heading-class',
        'id' => 'heading-1',
        'updatedAt' => '2024-01-01',
      ],
      'content' => [
        ['type' => 'text', 'text' => 'Title'],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($heading);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    // Only level should be kept for headings.
    $this->assertArrayHasKey('attrs', $data);
    $this->assertArrayHasKey('level', $data['attrs']);
    $this->assertEquals(2, $data['attrs']['level']);
    $this->assertCount(1, $data['attrs']);
  }

  /**
   * Tests media node attribute whitelisting.
   */
  public function testMediaNodeAttributes(): void {
    $media = [
      'type' => 'media',
      'attrs' => [
        'data-entity-type' => 'media',
        'data-entity-uuid' => '550e8400-e29b-41d4-a716-446655440000',
        'alt' => 'Alternative text',
        'title' => 'Title text',
        'width' => 100,
        'height' => 200,
        'custom' => 'value',
        'updatedAt' => '2024-01-01',
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($media);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    // Only entity type and UUID should be kept.
    $this->assertArrayHasKey('attrs', $data);
    $this->assertArrayHasKey('data-entity-type', $data['attrs']);
    $this->assertArrayHasKey('data-entity-uuid', $data['attrs']);
    $this->assertCount(2, $data['attrs']);
  }

  /**
   * Tests icon node attribute whitelisting.
   */
  public function testIconNodeAttributes(): void {
    $icon = [
      'type' => 'icon',
      'attrs' => [
        'iconName' => 'fa-home',
        'size' => 'large',
        'color' => 'red',
        'custom' => 'value',
        'updatedAt' => '2024-01-01',
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($icon);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    // Only iconName should be kept.
    $this->assertArrayHasKey('attrs', $data);
    $this->assertArrayHasKey('iconName', $data['attrs']);
    $this->assertEquals('fa-home', $data['attrs']['iconName']);
    $this->assertCount(1, $data['attrs']);
  }

  /**
   * Tests table cell attribute whitelisting.
   */
  public function testTableCellAttributes(): void {
    $table = [
      'type' => 'table',
      'content' => [
        [
          'type' => 'table_row',
          'content' => [
            [
              'type' => 'table_cell',
              'attrs' => [
                'colspan' => 2,
                'rowspan' => 3,
                'colwidth' => 100,
                'align' => 'center',
                'background' => 'red',
                'custom' => 'value',
                'updatedAt' => '2024-01-01',
              ],
              'content' => [
                [
                  'type' => 'paragraph',
                  'content' => [
                    ['type' => 'text', 'text' => 'Cell'],
                  ],
                ],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($table);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    $cellAttrs = $data['content'][0]['content'][0]['attrs'];
    // Only colspan, rowspan, and colwidth should be kept.
    $this->assertArrayHasKey('colspan', $cellAttrs);
    $this->assertArrayHasKey('rowspan', $cellAttrs);
    $this->assertArrayHasKey('colwidth', $cellAttrs);
    $this->assertCount(3, $cellAttrs);
  }

  /**
   * Tests text node attribute whitelisting.
   */
  public function testTextNodeAttributes(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Hello',
          'attrs' => [
            'custom' => 'value',
            'style' => 'color: red',
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    $textNode = $data['content'][0];
    // Text nodes should not have attrs.
    $this->assertArrayNotHasKey('attrs', $textNode);
  }

  /**
   * Tests bold mark attribute whitelisting.
   */
  public function testBoldMarkAttributes(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Bold text',
          'marks' => [
            [
              'type' => 'bold',
              'attrs' => [
                'style' => 'font-weight: 900',
                'custom' => 'value',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    $mark = $data['content'][0]['marks'][0];
    // Bold marks should not have attrs.
    $this->assertArrayNotHasKey('attrs', $mark);
  }

  /**
   * Tests link mark attribute whitelisting - external links.
   */
  public function testExternalLinkMarkAttributes(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link text',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'href' => 'https://example.com',
                'linkType' => 'external',
                'title' => 'Example',
                'target' => '_blank',
                'rel' => 'noopener',
                'class' => 'external-link',
                'custom' => 'value',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    $linkAttrs = $data['content'][0]['marks'][0]['attrs'];
    // Only href, linkUri, and linkType should be kept for external links.
    $this->assertArrayHasKey('href', $linkAttrs);
    $this->assertArrayHasKey('linkUri', $linkAttrs);
    $this->assertArrayHasKey('linkType', $linkAttrs);
    $this->assertEquals('external', $linkAttrs['linkType']);
    $this->assertCount(3, $linkAttrs);
  }

  /**
   * Tests link mark attribute whitelisting - internal links.
   */
  public function testInternalLinkMarkAttributes(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link text',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'href' => 'entity:node/550e8400-e29b-41d4-a716-446655440000',
                'linkType' => 'internal',
                'entityType' => 'node',
                'entityUuid' => '550e8400-e29b-41d4-a716-446655440000',
                'entityLabel' => 'My Node',
                'variant' => 'button',
                'class' => 'internal-link',
                'custom' => 'value',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    $linkAttrs = $data['content'][0]['marks'][0]['attrs'];
    // Only specific attributes should be kept for internal links.
    $this->assertArrayHasKey('href', $linkAttrs);
    $this->assertArrayHasKey('linkType', $linkAttrs);
    $this->assertArrayHasKey('entityType', $linkAttrs);
    $this->assertArrayHasKey('entityUuid', $linkAttrs);
    $this->assertArrayHasKey('variant', $linkAttrs);
    $this->assertEquals('internal', $linkAttrs['linkType']);
    $this->assertArrayNotHasKey('entityLabel', $linkAttrs);
    $this->assertArrayNotHasKey('class', $linkAttrs);
    $this->assertArrayNotHasKey('custom', $linkAttrs);
  }

  /**
   * Tests list node attribute whitelisting.
   */
  public function testListNodeAttributes(): void {
    // Test bullet list.
    $bulletList = [
      'type' => 'bullet_list',
      'attrs' => [
        'tight' => TRUE,
        'custom' => 'value',
        'class' => 'my-list',
      ],
      'content' => [
        [
          'type' => 'list_item',
          'content' => [
            [
              'type' => 'paragraph',
              'content' => [
                ['type' => 'text', 'text' => 'Item'],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($bulletList);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    // Bullet lists should not have attrs.
    $this->assertArrayNotHasKey('attrs', $data);

    // Test ordered list.
    $orderedList = [
      'type' => 'ordered_list',
      'attrs' => [
        'order' => 2,
        'tight' => TRUE,
        'custom' => 'value',
        'class' => 'my-list',
      ],
      'content' => [
        [
          'type' => 'list_item',
          'content' => [
            [
              'type' => 'paragraph',
              'content' => [
                ['type' => 'text', 'text' => 'Item'],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($orderedList);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    // Ordered lists should not have attrs in our system.
    $this->assertArrayNotHasKey('attrs', $data);
  }

  /**
   * Tests blockquote node attribute whitelisting.
   */
  public function testBlockquoteNodeAttributes(): void {
    $blockquote = [
      'type' => 'blockquote',
      'attrs' => [
        'cite' => 'https://example.com',
        'custom' => 'value',
        'class' => 'my-quote',
      ],
      'content' => [
        [
          'type' => 'paragraph',
          'content' => [
            ['type' => 'text', 'text' => 'Quote'],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($blockquote);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    // Blockquotes should not have attrs in our system.
    $this->assertArrayNotHasKey('attrs', $data);
  }

  /**
   * Tests horizontal_rule node attribute whitelisting.
   */
  public function testHorizontalRuleAttributes(): void {
    $hr = [
      'type' => 'horizontal_rule',
      'attrs' => [
        'class' => 'divider',
        'style' => 'border-top: 2px solid red',
        'custom' => 'value',
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($hr);
    $this->assertTrue($result->isValid());

    $data = $result->getData();
    // HR should not have attrs.
    $this->assertArrayNotHasKey('attrs', $data);
  }

}
