<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Transformation;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\prosemirror\Element\ElementProvider;
use Drupal\prosemirror\Entity\ProseMirrorElement;
use Drupal\prosemirror\Entity\ProseMirrorMark;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Drupal\prosemirror\Transformation\TransformationHelper;
use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\prosemirror\Transformation\EntityReference;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;
use Psr\Log\LoggerInterface;
use Drupal\Tests\prosemirror\Unit\ProseMirrorElementTestTrait;
use Drupal\Core\Entity\EntityRepositoryInterface;

/**
 * Tests the EntityReference class.
 *
 * @group prosemirror
 * @coversDefaultClass \Drupal\prosemirror\Transformation\EntityReference
 */
class EntityReferenceTest extends UnitTestCase {

  use ProphecyTrait;
  use ProseMirrorElementTestTrait;

  /**
   * Mock entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    [, $this->entityRepository] = $this->createMockEntityServices();

    // Create service container with element provider and entity repository.
    $container = new ContainerBuilder();
    $container->set('entity.repository', $this->entityRepository->reveal());
    \Drupal::setContainer($container);
  }

  /**
   * Tests constructor and getters.
   *
   * @covers ::__construct
   * @covers ::getEntityType
   * @covers ::getEntityUuid
   */
  public function testConstructorAndGetters(): void {
    $entityType = 'node';
    $entityUuid = '550e8400-e29b-41d4-a716-446655440000';

    $reference = new EntityReference($entityType, $entityUuid);

    $this->assertEquals($entityType, $reference->getEntityType());
    $this->assertEquals($entityUuid, $reference->getEntityUuid());
  }

  /**
   * Tests toArray method.
   *
   * @covers ::toArray
   */
  public function testToArray(): void {
    $reference = new EntityReference('node', '550e8400-e29b-41d4-a716-446655440000');

    $expected = [
      'entity_type' => 'node',
      'entity_uuid' => '550e8400-e29b-41d4-a716-446655440000',
    ];

    $this->assertEquals($expected, $reference->toArray());
  }

  /**
   * Tests that entity references can be properly compared.
   */
  public function testEntityReferenceComparison(): void {
    $ref1 = new EntityReference('node', '550e8400-e29b-41d4-a716-446655440000');
    $ref2 = new EntityReference('node', '550e8400-e29b-41d4-a716-446655440000');
    $ref3 = new EntityReference('media', '550e8400-e29b-41d4-a716-446655440000');
    $ref4 = new EntityReference('node', '550e8400-e29b-41d4-a716-446655440001');

    // Same type and UUID should be considered equal.
    $this->assertEquals($ref1->getEntityType(), $ref2->getEntityType());
    $this->assertEquals($ref1->getEntityUuid(), $ref2->getEntityUuid());

    // Different type, same UUID.
    $this->assertNotEquals($ref1->getEntityType(), $ref3->getEntityType());
    $this->assertEquals($ref1->getEntityUuid(), $ref3->getEntityUuid());

    // Same type, different UUID.
    $this->assertEquals($ref1->getEntityType(), $ref4->getEntityType());
    $this->assertNotEquals($ref1->getEntityUuid(), $ref4->getEntityUuid());
  }

  /**
   * Tests isValid method.
   *
   * @covers ::isValid
   */
  public function testIsValid(): void {
    $reference = new EntityReference('node', self::VALID_UUIDS[0]);
    $this->assertTrue($reference->isValid($this->entityRepository->reveal()));

    // Test with invalid UUID.
    $invalidReference = new EntityReference('node', self::INVALID_UUIDS[0]);
    $this->assertFalse($invalidReference->isValid($this->entityRepository->reveal()));
  }

}
