<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Transformation;

use Drupal\Tests\UnitTestCase;
use Drupal\prosemirror\Transformation\TransformationHelper;
use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\prosemirror\Transformation\EntityReference;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Drupal\prosemirror\Element\ElementProvider;
use Drupal\prosemirror\Entity\ProseMirrorElement;
use Drupal\prosemirror\Entity\ProseMirrorMark;
use Psr\Log\LoggerInterface;
use Prophecy\Prophecy\ObjectProphecy;
use Prophecy\Argument;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Tests link validation for TransformationHelper.
 *
 * @coversDefaultClass \Drupal\prosemirror\Transformation\TransformationHelper
 * @group prosemirror
 */
class LinkValidationTest extends UnitTestCase {

  use \Drupal\Tests\prosemirror\Unit\ProseMirrorElementTestTrait;

  /**
   * The transformation helper service.
   *
   * @var \Drupal\prosemirror\Transformation\TransformationHelper
   */
  protected TransformationHelper $transformationHelper;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityTypeManager;

  /**
   * Mock entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityRepository;

  /**
   * Mock element type manager.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementTypeManager;

  /**
   * Mock element provider.
   *
   * @var \Drupal\prosemirror\Element\ElementProvider|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementProvider;

  /**
   * Mock logger.
   *
   * @var \Psr\Log\LoggerInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    [$this->entityTypeManager, $this->entityRepository] = $this->createMockEntityServices();
    $this->elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $this->logger = $this->prophesize(LoggerInterface::class);
    $this->elementProvider = $this->prophesize(ElementProvider::class);

    // Setup elements and marks using trait.
    $this->setupElementProviderMock($this->elementProvider);

    // Create service container.
    $container = new ContainerBuilder();
    $container->set('prosemirror.element_provider', $this->elementProvider->reveal());
    $container->set('entity.repository', $this->entityRepository->reveal());
    \Drupal::setContainer($container);

    // Create transformation helper with mocked dependencies.
    $this->transformationHelper = new TransformationHelper(
      $this->entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $this->elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $this->logger->reveal()
    );
  }

  /**
   * Tests link mark missing required linkType attribute.
   */
  public function testLinkMarkMissingLinkType(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'href' => 'https://example.com',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Link mark missing required linkType attribute', $errors[0]->getMessage());
  }

  /**
   * Tests link mark missing required href attribute.
   */
  public function testLinkMarkMissingHref(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'linkType' => 'external',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Link mark missing required href attribute', $errors[0]->getMessage());
  }

  /**
   * Tests link mark with invalid entity type.
   */
  public function testLinkMarkInvalidEntityType(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'linkType' => 'internal',
                'href' => 'entity:invalid_type/550e8400-e29b-41d4-a716-446655440000',
                'entityType' => 'invalid_type',
                'entityUuid' => '550e8400-e29b-41d4-a716-446655440000',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Invalid entity type for link mark', $errors[0]->getMessage());
  }

  /**
   * Tests link mark with invalid entity UUID.
   */
  public function testLinkMarkInvalidEntityUuid(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'linkType' => 'internal',
                'href' => 'entity:node/invalid-uuid',
                'entityType' => 'node',
                'entityUuid' => 'invalid-uuid',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Invalid content reference', $errors[0]->getMessage());
  }

  /**
   * Tests internal link missing entity attributes.
   */
  public function testInternalLinkMissingEntityAttributes(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'linkType' => 'internal',
                'href' => 'https://example.com',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Link mark missing required entityUuid and entityType attributes', $errors[0]->getMessage());
  }

  /**
   * Tests link mark with invalid href format.
   */
  public function testLinkMarkInvalidHrefFormat(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'linkType' => 'external',
                'href' => 'not-a-valid-url',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Invalid link href format', $errors[0]->getMessage());
  }

  /**
   * Tests valid external link.
   */
  public function testValidExternalLink(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'linkType' => 'external',
                'href' => 'https://example.com',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertTrue($result->isValid());
    $this->assertEmpty($result->getErrors());
  }

  /**
   * Tests valid internal link.
   */
  public function testValidInternalLink(): void {
    $paragraph = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'linkType' => 'internal',
                'href' => 'entity:node/550e8400-e29b-41d4-a716-446655440000',
                'entityType' => 'node',
                'entityUuid' => '550e8400-e29b-41d4-a716-446655440000',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraph);
    $this->assertTrue($result->isValid());
    $this->assertEmpty($result->getErrors());
    $references = $result->getReferences();
    $this->assertCount(1, $references);
    $this->assertEquals('node', $references[0]->getEntityType());
    $this->assertEquals('550e8400-e29b-41d4-a716-446655440000', $references[0]->getEntityUuid());
  }

}
