<?php

declare(strict_types=1);

namespace Drupal\Tests\prosemirror\Unit\Transformation;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\prosemirror\Element\ElementProvider;
use Drupal\prosemirror\Entity\ProseMirrorElement;
use Drupal\prosemirror\Entity\ProseMirrorMark;
use Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager;
use Drupal\prosemirror\Transformation\TransformationHelper;
use Drupal\prosemirror\Transformation\ValidationError;
use Drupal\prosemirror\Transformation\EntityReference;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;
use Prophecy\Argument;
use Psr\Log\LoggerInterface;
use Drupal\Tests\prosemirror\Unit\ProseMirrorElementTestTrait;
use Drupal\Core\Entity\EntityRepositoryInterface;

/**
 * Tests the TransformationHelper class.
 *
 * @group prosemirror
 * @coversDefaultClass \Drupal\prosemirror\Transformation\TransformationHelper
 */
class TransformationHelperTest extends UnitTestCase {

  use ProphecyTrait;
  use ProseMirrorElementTestTrait;

  /**
   * The transformation helper service.
   *
   * @var \Drupal\prosemirror\Transformation\TransformationHelper
   */
  protected TransformationHelper $transformationHelper;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityTypeManager;

  /**
   * Mock entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $entityRepository;

  /**
   * Mock element type manager.
   *
   * @var \Drupal\prosemirror\Plugin\ProseMirrorElementTypeManager|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementTypeManager;

  /**
   * Mock logger.
   *
   * @var \Psr\Log\LoggerInterface|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $logger;

  /**
   * Mock element provider.
   *
   * @var \Drupal\prosemirror\Element\ElementProvider|\Prophecy\Prophecy\ObjectProphecy
   */
  protected $elementProvider;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    [$this->entityTypeManager, $this->entityRepository] = $this->createMockEntityServices();
    $this->elementTypeManager = $this->prophesize(ProseMirrorElementTypeManager::class);
    $this->logger = $this->prophesize(LoggerInterface::class);
    $this->elementProvider = $this->prophesize(ElementProvider::class);

    // Create service container with element provider and entity repository.
    $container = new ContainerBuilder();
    $container->set('prosemirror.element_provider', $this->elementProvider->reveal());
    $container->set('entity.repository', $this->entityRepository->reveal());
    \Drupal::setContainer($container);

    // Create transformation helper with mocked dependencies.
    $this->transformationHelper = new TransformationHelper(
      $this->entityTypeManager->reveal(),
      $this->entityRepository->reveal(),
      $this->elementTypeManager->reveal(),
      $this->elementProvider->reveal(),
      $this->logger->reveal()
    );
  }

  /**
   * Helper to set up element mocks.
   */
  protected function setupElementMocks(array $elements): void {
    $mockElements = [];
    foreach ($elements as $id => $type) {
      $element = $this->prophesize(ProseMirrorElement::class);
      $element->id()->willReturn($id);
      $element->getType()->willReturn($type);
      $element->getOptions()->willReturn([]);
      $element->getContentMin()->willReturn(NULL);
      $element->getContentMax()->willReturn(NULL);
      $mockElements[$id] = $element->reveal();
    }

    $this->elementProvider->getAllElements()
      ->willReturn($mockElements);

    // Mock that system elements don't have plugins.
    $this->elementTypeManager->createInstance('system', Argument::any())
      ->willThrow(new \Exception('No plugin'));
  }

  /**
   * Helper to set up mark mocks.
   */
  protected function setupMarkMocks(array $marks): void {
    $mockMarks = [];
    foreach ($marks as $id => $attrs) {
      $mark = $this->prophesize(ProseMirrorMark::class);
      $mark->id()->willReturn($id);
      $mark->getAttributes()->willReturn($attrs);
      $mockMarks[$id] = $mark->reveal();
    }

    $this->elementProvider->getAllMarks()
      ->willReturn($mockMarks);
  }

  /**
   * Tests validation of a paragraph element.
   *
   * @covers ::validateAndSanitize
   * @covers ::validateChildNode
   */
  public function testParagraphValidation(): void {
    $this->setupElementMocks([
      'paragraph' => 'system',
      'text' => 'system',
    ]);

    // Test valid paragraph.
    $validParagraph = [
      'type' => 'paragraph',
      'content' => [
        ['type' => 'text', 'text' => 'Hello world'],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($validParagraph);
    $this->assertTrue($result->isValid());
    $this->assertEmpty($result->getErrors());
    $this->assertEquals($validParagraph, $result->getData());

    // Test paragraph with invalid attributes (should be stripped)
    $paragraphWithInvalidAttrs = [
      'type' => 'paragraph',
      'attrs' => [
        'updatedAt' => '2024-01-01',
        'editDialog' => TRUE,
        'index' => 1,
      ],
      'content' => [
        ['type' => 'text', 'text' => 'Hello world'],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($paragraphWithInvalidAttrs);
    $this->assertTrue($result->isValid());
    $this->assertEmpty($result->getErrors());
    $data = $result->getData();
    if (isset($data['attrs'])) {
      $this->assertEmpty($data['attrs']);
    }
  }

  /**
   * Tests validation of heading elements.
   *
   * @covers ::validateAndSanitize
   */
  public function testHeadingValidation(): void {
    $this->setupElementMocks([
      'heading' => 'system',
      'text' => 'system',
    ]);

    // Test valid heading.
    $validHeading = [
      'type' => 'heading',
      'attrs' => ['level' => 2],
      'content' => [
        ['type' => 'text', 'text' => 'Section Title'],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($validHeading);
    $this->assertTrue($result->isValid());
    $this->assertEmpty($result->getErrors());
    // Expect equal except for the level attribute being stripped.
    $data = $result->getData();
    if (isset($data['attrs'])) {
      $this->assertEmpty($data['attrs']);
      // Everything else should be the same.
      $this->assertEquals($validHeading['content'], $data['content']);
      $this->assertEquals($validHeading['type'], $data['type']);
    }
    else {
      $this->assertEquals($validHeading, $result->getData());
    }
  }

  /**
   * Tests validation of media elements.
   *
   * @covers ::validateAndSanitize
   */
  public function testMediaValidation(): void {
    $this->setupElementMocks([
      'media' => 'system',
    ]);

    // Test valid media.
    $validMedia = [
      'type' => 'media',
      'attrs' => [
        'data-entity-type' => 'media',
        'data-entity-uuid' => self::VALID_UUIDS[0],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($validMedia);
    $this->assertEmpty($result->getErrors());
    $this->assertTrue($result->isValid());
    $references = $result->getReferences();
    $this->assertCount(1, $references);
    $this->assertEquals('media', $references[0]->getEntityType());
    $this->assertEquals(self::VALID_UUIDS[0], $references[0]->getEntityUuid());

    // Test media with missing UUID.
    $invalidMedia = [
      'type' => 'media',
      'attrs' => [
        'data-entity-type' => 'media',
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($invalidMedia);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('missing required entity-type or entity-uuid', $errors[0]->getMessage());

    // Test media with invalid UUID.
    $invalidUuidMedia = [
      'type' => 'media',
      'attrs' => [
        'data-entity-type' => 'media',
        'data-entity-uuid' => self::INVALID_UUIDS[0],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($invalidUuidMedia);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Invalid media reference', $errors[0]->getMessage());
  }

  /**
   * Tests validation of list elements.
   *
   * @covers ::validateAndSanitize
   */
  public function testListValidation(): void {
    $this->setupElementMocks([
      'bullet_list' => 'system',
      'ordered_list' => 'system',
      'list_item' => 'system',
      'paragraph' => 'system',
      'text' => 'system',
    ]);

    // Test valid bullet list.
    $validBulletList = [
      'type' => 'bullet_list',
      'content' => [
        [
          'type' => 'list_item',
          'content' => [
            [
              'type' => 'paragraph',
              'content' => [
                ['type' => 'text', 'text' => 'Item 1'],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($validBulletList);
    $this->assertTrue($result->isValid());
    $this->assertEmpty($result->getErrors());

    // Test valid ordered list.
    $validOrderedList = [
      'type' => 'ordered_list',
      'content' => [
        [
          'type' => 'list_item',
          'content' => [
            [
              'type' => 'paragraph',
              'content' => [
                ['type' => 'text', 'text' => 'Item 1'],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($validOrderedList);
    $this->assertTrue($result->isValid());
    $this->assertEmpty($result->getErrors());
  }

  /**
   * Tests validation of table elements.
   *
   * @covers ::validateAndSanitize
   */
  public function testTableValidation(): void {
    $this->setupElementMocks([
      'table' => 'system',
      'table_row' => 'system',
      'table_cell' => 'system',
      'table_header' => 'system',
      'paragraph' => 'system',
      'text' => 'system',
    ]);

    // Test valid table.
    $validTable = [
      'type' => 'table',
      'content' => [
        [
          'type' => 'table_row',
          'content' => [
            [
              'type' => 'table_cell',
              'attrs' => ['colspan' => 1, 'rowspan' => 1],
              'content' => [
                [
                  'type' => 'paragraph',
                  'content' => [
                    ['type' => 'text', 'text' => 'Cell content'],
                  ],
                ],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($validTable);
    $this->assertTrue($result->isValid());
    $this->assertEmpty($result->getErrors());

    // Test table cell with invalid colspan.
    $invalidColspanTable = [
      'type' => 'table',
      'content' => [
        [
          'type' => 'table_row',
          'content' => [
            [
              'type' => 'table_cell',
              'attrs' => ['colspan' => 'invalid'],
              'content' => [
                [
                  'type' => 'paragraph',
                  'content' => [
                    ['type' => 'text', 'text' => 'Cell content'],
                  ],
                ],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($invalidColspanTable);
    // Should still be valid, but invalid attributes are stripped.
    $this->assertTrue($result->isValid());
  }

  /**
   * Tests validation of marks.
   *
   * @covers ::sanitizeMarks
   */
  public function testMarkValidation(): void {
    $this->setupElementMocks([
      'paragraph' => 'system',
      'text' => 'system',
    ]);

    $this->setupMarkMocks([
      'bold' => [],
      'italic' => [],
      'link' => [
        'href' => ['type' => 'string'],
        'title' => ['type' => 'string'],
      ],
    ]);

    // Test valid marks.
    $validTextWithMarks = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Bold and italic text',
          'marks' => [
            ['type' => 'bold'],
            ['type' => 'italic'],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($validTextWithMarks);
    $this->assertEmpty($result->getErrors());
    $this->assertTrue($result->isValid());

    // Test valid external link.
    $validExternalLink = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link text',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'href' => 'https://example.com',
                'linkType' => 'external',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($validExternalLink);
    $this->assertTrue($result->isValid());
    $this->assertEmpty($result->getErrors());

    // Test valid internal link.
    $validInternalLink = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link text',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'href' => 'entity:node/' . self::VALID_UUIDS[0],
                'linkType' => 'internal',
                'entityType' => 'node',
                'entityUuid' => self::VALID_UUIDS[0],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($validInternalLink);
    $this->assertEmpty($result->getErrors());
    $this->assertTrue($result->isValid());
    $references = $result->getReferences();
    $this->assertCount(1, $references);
    $this->assertEquals('node', $references[0]->getEntityType());
    $this->assertEquals(self::VALID_UUIDS[0], $references[0]->getEntityUuid());

    // Test link without linkType.
    $linkWithoutType = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link text',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'href' => 'https://example.com',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($linkWithoutType);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('missing required linkType', $errors[0]->getMessage());

    // Test link without href.
    $linkWithoutHref = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link text',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'linkType' => 'external',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($linkWithoutHref);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('missing required href', $errors[0]->getMessage());

    // Test internal link without entity info.
    $internalLinkWithoutEntity = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Link text',
          'marks' => [
            [
              'type' => 'link',
              'attrs' => [
                'href' => 'entity:node/' . self::VALID_UUIDS[0],
                'linkType' => 'internal',
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($internalLinkWithoutEntity);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('missing required entityUuid and entityType', $errors[0]->getMessage());

    // Test unknown mark type.
    $unknownMark = [
      'type' => 'paragraph',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Text',
          'marks' => [
            ['type' => 'unknown_mark'],
          ],
        ],
      ],
    ];

    // Update mark storage mock to return empty for unknown mark.
    $this->elementProvider->getAllMarks()
      ->willReturn([]);

    $result = $this->transformationHelper->validateAndSanitize($unknownMark);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Unknown mark type', $errors[0]->getMessage());
  }

  /**
   * Tests handling of missing type property.
   *
   * @covers ::validateAndSanitize
   */
  public function testMissingTypeProperty(): void {
    $nodeWithoutType = [
      'content' => [
        ['type' => 'text', 'text' => 'Hello'],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($nodeWithoutType);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('missing required "type" property', $errors[0]->getMessage());
  }

  /**
   * Tests handling of unknown node type.
   *
   * @covers ::validateAndSanitize
   */
  public function testUnknownNodeType(): void {
    // Return empty array for unknown elements.
    $this->elementProvider->getAllElements()
      ->willReturn([]);

    $unknownNode = [
      'type' => 'unknown_type',
      'content' => [
        ['type' => 'text', 'text' => 'Hello'],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($unknownNode);
    $this->assertFalse($result->isValid());
    $errors = $result->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString('Unknown node type: unknown_type', $errors[0]->getMessage());
  }

  /**
   * Tests complex nested document structure.
   *
   * @covers ::validateAndSanitize
   */
  public function testComplexDocument(): void {
    $this->setupElementMocks([
      'doc' => 'system',
      'paragraph' => 'system',
      'heading' => 'system',
      'bullet_list' => 'system',
      'list_item' => 'system',
      'blockquote' => 'system',
      'horizontal_rule' => 'system',
      'text' => 'system',
    ]);

    $this->setupMarkMocks([
      'bold' => [],
      'italic' => [],
    ]);

    // Complex valid document.
    $complexDoc = [
      'type' => 'doc',
      'content' => [
        [
          'type' => 'heading',
          'attrs' => ['level' => 1],
          'content' => [
            ['type' => 'text', 'text' => 'Main Title'],
          ],
        ],
        [
          'type' => 'paragraph',
          'content' => [
            [
              'type' => 'text',
              'text' => 'This is ',
            ],
            [
              'type' => 'text',
              'text' => 'bold',
              'marks' => [['type' => 'bold']],
            ],
            [
              'type' => 'text',
              'text' => ' and ',
            ],
            [
              'type' => 'text',
              'text' => 'italic',
              'marks' => [['type' => 'italic']],
            ],
            [
              'type' => 'text',
              'text' => ' text.',
            ],
          ],
        ],
        [
          'type' => 'bullet_list',
          'content' => [
            [
              'type' => 'list_item',
              'content' => [
                [
                  'type' => 'paragraph',
                  'content' => [
                    ['type' => 'text', 'text' => 'First item'],
                  ],
                ],
              ],
            ],
            [
              'type' => 'list_item',
              'content' => [
                [
                  'type' => 'paragraph',
                  'content' => [
                    ['type' => 'text', 'text' => 'Second item'],
                  ],
                ],
              ],
            ],
          ],
        ],
        ['type' => 'horizontal_rule'],
        [
          'type' => 'blockquote',
          'content' => [
            [
              'type' => 'paragraph',
              'content' => [
                ['type' => 'text', 'text' => 'This is a quote.'],
              ],
            ],
          ],
        ],
      ],
    ];

    $result = $this->transformationHelper->validateAndSanitize($complexDoc);
    $this->assertEmpty($result->getErrors());
    $this->assertTrue($result->isValid());
  }

}
