import { library, icon, IconName } from '@fortawesome/fontawesome-svg-core'
import { fas } from '@fortawesome/free-solid-svg-icons'

// Types are available through the main ProseMirror module's global API
type NodeSpec = any;
type DOMOutputSpec = any;
type Plugin = any;
type MenuItem = any;
type Schema = any;

// Add all solid icons to the library
library.add(fas);

// Extend Drupal.ProsemirrorTemplateTypes with icon template
declare global {
  interface Window {
    Drupal: {
      ProsemirrorTemplateTypes: any;
      ProseMirrorAPI?: {
        modules: {
          'prosemirror-model': any;
          'prosemirror-state': any;
          'prosemirror-menu': any;
        };
        registerNodePlugin: (plugin: any) => void;
        registerMarkPlugin: (plugin: any) => void;
        registerMenuExtension: (extension: any) => void;
        registerPluginExtension: (extension: any) => void;
      };
    };
  }
}

// Check that Drupal and ProsemirrorTemplateTypes are available

if (!window.Drupal || !window.Drupal.ProsemirrorTemplateTypes) {
  console.error('Drupal.ProsemirrorTemplateTypes not available when FontAwesome module loaded');
  throw new Error('Drupal.ProsemirrorTemplateTypes not available. Ensure prosemirror/editor library is loaded before prosemirror_fontawesome_icons/fontawesome-icons.');
}

registerIconTemplate();

export interface IconConfig {
  name: string;
  displayName: string;
  iconName: string;
  nodeType: 'icon';
}

function registerIconTemplate() {
  //console.debug('FontAwesome: Registering icon template...');

  // Helper function to convert camelCase to kebab-case
  function camelToKebab(str: string): string {
    return str
      .replace(/^fa/, '') // Remove 'fa' prefix
      .replace(/([a-z0-9])([A-Z])/g, '$1-$2') // Add dash before capital letters
      .toLowerCase(); // Convert to lowercase
  }

  // Icon picker dialog
  function openIconPicker(callback: (iconName: string) => void, appWrapper: any) {
    const renderer = appWrapper.getRendererFactory();

    // Create dialog content
    const dialogContentResult = renderer.elements.div({ className: 'prosemirror-fontawesome-icons-dialog-picker-dialog-content' });
    const dialogContent = dialogContentResult.element;

    const searchInputResult = renderer.elements.input({
      type: 'text',
      placeholder: 'Search icons...',
      className: 'icon-search-input'
    });
    const searchInput = searchInputResult.element;

    const iconGridResult = renderer.elements.div({ className: 'icon-grid' });
    const iconGrid = iconGridResult.element;

    dialogContent.appendChild(searchInput);
    dialogContent.appendChild(iconGrid);

    function updateIcons(filter = '') {
      iconGrid.innerHTML = '';

      // Store unique icons by their raw icon object
      const uniqueIcons = new Map();

      // First pass: collect unique icons using raw fas objects
      Object.entries(fas).forEach(([camelName, iconObj]) => {
        if (!uniqueIcons.has(iconObj)) {
          uniqueIcons.set(iconObj, camelName);
        }
      });

      // Second pass: display unique icons
      Array.from(uniqueIcons.values())
        .map((camelName: string) => camelToKebab(camelName) as IconName)
        .filter((name: string) => name.toLowerCase().includes(filter.toLowerCase()))
        .sort()
        .forEach((iconName: IconName) => {
          try {
            const iconResult = icon({ prefix: 'fas', iconName: iconName });
            if (iconResult && iconResult.node && iconResult.node[0]) {
              const iconWrapperResult = renderer.elements.div({
                className: 'icon-option',
                attributes: { title: iconName },
                listeners: {
                  click: () => {
                    callback(iconName);
                    dialog.close();
                  }
                }
              });
              const iconWrapper = iconWrapperResult.element;
              iconWrapper.appendChild(iconResult.node[0]);
              iconGrid.appendChild(iconWrapper);
            }
          } catch (error) {
            // Silently skip invalid icons
          }
        });
    }

    searchInput.addEventListener('input', (e: Event) => {
      updateIcons((e.target as HTMLInputElement).value);
    });

    updateIcons();

    // Create and open dialog using renderer
    const dialog = renderer.components.dialog({
      title: 'Select Icon',
      content: dialogContent,
      width: '600px',
      height: '400px',
      modal: true
    });

    dialog.open();
  }

  // Custom node view to handle icon rendering
  class FontAwesomeView {
    dom: HTMLElement;
    private appWrapper: any;

    constructor(node: any, appWrapper: any) {
      this.appWrapper = appWrapper;
      const renderer = appWrapper.getRendererFactory();
      const spanResult = renderer.elements.span({ className: 'font-awesome-icon' });
      this.dom = spanResult.element;
      this.update(node);
    }

    update(node: any) {
      const iconElement = icon({ prefix: 'fas', iconName: node.attrs.iconName }).node[0];
      this.dom.innerHTML = '';
      this.dom.appendChild(iconElement);
      return true;
    }

    destroy() {
      // No cleanup needed
    }
  }

  // Icon template function following the same pattern as base-node
  const iconTemplate = (spec: IconConfig, appWrapper: any) => {
    //console.debug('FontAwesome: Creating icon template for:', spec.name);

    // Get ProseMirror classes from global API (they should be available since prosemirror/editor is a dependency)
    const proseMirrorAPI = (window as any).Drupal?.ProseMirrorAPI;
    const Plugin = proseMirrorAPI?.modules?.['prosemirror-state']?.Plugin;
    const MenuItem = proseMirrorAPI?.modules?.['prosemirror-menu']?.MenuItem;

    // Create the DOM output spec
    const toDOM = (node: any): DOMOutputSpec => {
      const iconElement = icon({ prefix: 'fas', iconName: node.attrs.iconName }).node[0];
      return ["span", { class: "font-awesome-icon", "data-icon": node.attrs.iconName }, iconElement];
    };

    // Build the node spec
    const nodeSpec: NodeSpec = {
      attrs: {
        iconName: { default: spec.iconName || 'camera' }
      },
      inline: true,
      group: "inline",
      draggable: true,
      toDOM,
      parseDOM: [{
        tag: "span.font-awesome-icon",
        getAttrs: (dom: HTMLElement) => {
          const iconName = dom.getAttribute('data-icon') || 'camera';
          return { iconName };
        }
      }]
    };

    // Create menu item
    const menuItem = (schema: any) => {
      if (!MenuItem) {
        console.warn('FontAwesome: MenuItem class not available');
        return null;
      }

      return new MenuItem({
        title: `Insert ${spec.displayName}`,
        label: spec.displayName,
        enable(state: any) {
          // For inline nodes, check if we can insert at cursor
          return state.selection.$from.parent.inlineContent;
        },
        run(state: any, dispatch: any) {
          //console.debug('FontAwesome: Icon menu item clicked');
          openIconPicker((iconName: string) => {
            const node = schema.nodes[spec.name].create({ iconName });
            dispatch?.(state.tr.replaceSelectionWith(node));
          }, appWrapper);
        }
      });
    };

    // Plugin for custom node view
    const plugin = Plugin ? new Plugin({
      props: {
        nodeViews: {
          [spec.name]: (node: any) => new FontAwesomeView(node, appWrapper)
        }
      }
    }) : null;

    return {
      name: spec.name,
      spec: nodeSpec,
      plugin,
      menuName: 'insert',
      menuItem,
    };
  };

  // Register the icon template type
  window.Drupal.ProsemirrorTemplateTypes.icon = iconTemplate;

  //console.debug('FontAwesome: Icon template registered successfully!');
}
