<?php

namespace Drupal\protect_views_flood_control\Plugin\views\display_extender;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\views\Attribute\ViewsDisplayExtender;
use Drupal\views\Plugin\views\display_extender\DisplayExtenderPluginBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Protect Views Flood Control display extender plugin.
 *
 * @ingroup views_display_extender_plugins
 */
#[ViewsDisplayExtender(
  id: 'protect_views_flood_control',
  title: new TranslatableMarkup('Protect Views Flood Control'),
  help: new TranslatableMarkup('Restricts the number of Views Exposed Filter form submissions a user can make in a given timeframe.'),
  no_ui: FALSE,
)]
class ProtectViewsFloodControl extends DisplayExtenderPluginBase {

  /**
   * Add an item in the Advanced panel that opens our form section.
   */
  public function optionsSummary(&$categories, &$options) {
    $categories['flood_control'] = [
      'title' => $this->t('Flood control'),
      'column' => 'second',
    ];

    $options['protect_views_flood_control'] = [
      'category' => 'flood_control',
      'title' => $this->t('Flood control'),
      'value' => $this->t('Disabled'),
      'desc' => $this->t('Limit exposed form submissions via flood control.'),
    ];
    if (isset($this->options['enable_protect_views_flood_control']) && $this->options['enable_protect_views_flood_control']) {
      $options['protect_views_flood_control']['value'] = $this->t('Enabled');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    if ($form_state->get('section') !== 'protect_views_flood_control') {
      return;
    }

    $form['enable_protect_views_flood_control'] = [
      '#title' => $this->t('Protect Views Exposed Form with Flood Control'),
      '#type' => 'checkbox',
      '#description' => $this->t('Enables flood control on the Views Exposed Form for this display.'),
      '#default_value' => $this->options['enable_protect_views_flood_control'] ?? 0,
    ];

    $form['protect_views_flood_control_window'] = [
      '#title' => $this->t('Window'),
      '#type' => 'number',
      '#description' => $this->t('The timeframe in seconds that views exposed form will be tracked for a given IP address.'),
      '#default_value' => $this->options['protect_views_flood_control_window'] ?? 30,
      '#min' => 10,
      '#size' => 5,
      '#states' => [
        'visible' => [
          ':input[name="enable_protect_views_flood_control"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['protect_views_flood_control_threshold'] = [
      '#title' => $this->t('Threshold'),
      '#type' => 'number',
      '#description' => $this->t('The number of submissions allowed in the given timeframe by the given IP address.'),
      '#default_value' => $this->options['protect_views_flood_control_threshold'] ?? 5,
      '#min' => 1,
      '#size' => 5,
      '#states' => [
        'visible' => [
          ':input[name="enable_protect_views_flood_control"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['markup'] = [
      '#prefix' => '<div class="js-form-item form-item description">',
      '#markup' => $this->t('This applies when a user changes filters only, not on initial load, nor when changing pages. If the result is cached and views exposed form is not put to work, this is not included. To white list IP addresses and control logging, see the <a href="@link_to_settings">Protect Form Flood Control settings</a>.', [
        '@link_to_settings' => Url::fromRoute('protect_form_flood_control.settings')->toString(),
      ]),
      '#suffix' => '</div>',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    $this->options['enable_protect_views_flood_control'] = (bool) $form_state->getValue('enable_protect_views_flood_control');
    $this->options['protect_views_flood_control_window'] = (int) $form_state->getValue('protect_views_flood_control_window');
    $this->options['protect_views_flood_control_threshold'] = (int) $form_state->getValue('protect_views_flood_control_threshold');
  }

}
