<?php

namespace Drupal\protect_views_flood_control\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\views\ViewExecutable;
use Symfony\Component\HttpKernel\Exception\TooManyRequestsHttpException;

/**
 * Class for hooks from the Protect Views Flood Control module.
 */
class ProtectViewsFloodControl {

  use StringTranslationTrait;

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match): ?string {
    switch ($route_name) {
      case 'help.page.protect_views_flood_control':
        $output = '';
        $output .= '<h3>' . $this->t('About') . '</h3>';
        $output .= '<p>' . $this->t('Edit your Views under Advanced to enable flood control protection for Views Exposed Forms.') . '</p>';
        return $output;

      default:
    }
    return NULL;
  }

  /**
   * Add our validator to the Views exposed form.
   *
   * Implements hook_form_FORM_ID_alter() for views_exposed_form.
   */
  #[Hook('form_views_exposed_form_alter')]
  public function formViewsExposedFormAlter(array &$form, FormStateInterface $form_state): void {
    $view = $form_state->get('view');
    if (!$view instanceof ViewExecutable) {
      return;
    }
    // Use a static callable so it survives form cache.
    $form['#validate'][] = [self::class, 'exposedValidate'];
  }

  /**
   * Validate handler that enforces flood control per view/display.
   *
   * NOTE: static so it can be safely stored in form cache.
   */
  public static function exposedValidate(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\views\ViewExecutable|null $view */
    $view = $form_state->get('view');
    if (!$view instanceof ViewExecutable) {
      return;
    }

    $display = $view->display_handler;

    // Find our display extender instance and read its options.
    foreach ($display->getExtenders() as $extender) {
      if ($extender->getPluginId() === 'protect_views_flood_control') {
        $options = $extender->options ?? [];
        if (self::shouldHandleFloodProtection($view, $options, $form, $form_state)) {
          self::handleFloodProtection($view, $options, $form, $form_state);
        }
        return;
      }
    }
  }

  /**
   * Check if flood protection should be applied to the exposed form.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view.
   * @param array $options
   *   The display extender option values.
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The view exposed form state.
   *
   * @return bool
   *   True if flood protection should be applied.
   */
  protected static function shouldHandleFloodProtection(ViewExecutable $view, array $options, array &$form, FormStateInterface $form_state): bool {
    // Only act if enabled for this display.
    if (empty($options['enable_protect_views_flood_control'])) {
      return FALSE;
    }

    // Only act if there are exposed input fields.
    $exposed_input = $view->getExposedInput();
    if (empty($exposed_input)) {
      return FALSE;
    }
    $exposed_input = array_filter($exposed_input, function ($value) {
      if (empty($value) || $value === 'All') {
        return FALSE;
      }
      return TRUE;
    });
    if (empty($exposed_input)) {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Handle flood protection for the exposed form.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view.
   * @param array $options
   *   The display extender option values.
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The view exposed form state.
   */
  protected static function handleFloodProtection(ViewExecutable $view, array $options, array &$form, FormStateInterface $form_state): void {
    $window = (int) ($options['protect_views_flood_control_window'] ?? 30);
    $threshold = (int) ($options['protect_views_flood_control_threshold'] ?? 5);

    // Separate flood control per View ID.
    $key = sprintf('views_exposed:%s:%s', $view->id(), $view->current_display);

    /** @var \Drupal\protect_form_flood_control\ManagerInterface $manager */
    $manager   = \Drupal::service('protect_form_flood_control.manager');
    $truncated = $manager->truncateFormId($key);
    $is_ajax   = \Drupal::request()->isXmlHttpRequest();
    if (!$manager->getFlood()->isAllowed($truncated, $threshold, $window)) {
      if ($is_ajax) {

        // Ajax request likely human instigated. Send a readable message back
        // to the user.
        $form_state->setError($form, t('You cannot submit the filters more than @threshold times in @window. Please try again later.', [
          '@threshold' => $threshold,
          '@window' => $manager->getDateFormatter()->formatInterval($window),
        ]));
        if ($manager->shouldLogBlockedSubmissions()) {
          $manager->logBlockedSubmission($key, $window, $threshold);
        }
      }
      else {

        // Http get request is likely AI scraping, we should nicely tell
        // the scraping bot to slow down and indicate how slow it should
        // go.
        $retry_after = max(1, $window);
        throw new TooManyRequestsHttpException(
          $retry_after,
          sprintf('Rate limit exceeded. Please try again in %s.', $manager->getDateFormatter()->formatInterval($window)),
          previous: NULL,
          code: 0,
          headers: [
            'X-RateLimit-Limit' => (string) $threshold,
          ],
        );
      }
    }
    else {

      // Record the exposed form usage in the flood table.
      $manager->getFlood()->register($truncated, $window);
    }
  }

}
