<?php

namespace Drupal\Tests\protect_views_flood_control\Functional;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\taxonomy\Traits\TaxonomyTestTrait;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\field\Traits\EntityReferenceFieldCreationTrait;
use Drupal\taxonomy\VocabularyInterface;

/**
 * The views flood control protection general tests.
 *
 * @group protect_views_flood_control
 */
class ProtectViewsFloodControlTest extends BrowserTestBase {

  use StringTranslationTrait;
  use TaxonomyTestTrait;
  use ContentTypeCreationTrait;
  use EntityReferenceFieldCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Enable only core and our module upfront.
   *
   * The test module (which ships the View) is installed later in setUp()
   * after we create the vocabulary/field it depends on.
   *
   * @var array
   */
  protected static $modules = [
    'user',
    'node',
    'views',
    'taxonomy',
    'field',
    'protect_form_flood_control',
    'protect_views_flood_control',
  ];

  /**
   * The vocabulary.
   *
   * @var \Drupal\taxonomy\VocabularyInterface
   */
  protected VocabularyInterface $vocab;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create the content type used by the View.
    $this->drupalCreateContentType([
      'type' => 'article',
      'name' => 'Article',
    ]);

    // Create the vocabulary and keep the object.
    $this->vocab = $this->createVocabulary([
      'name' => 'Tags',
      'vid' => 'tags',
    ]);

    // Add an entity reference field 'field_tags' on nodes:article ->
    // taxonomy_term:tags.
    $this->createEntityReferenceField(
      'node',
      'article',
      'field_tags',
      'Tags',
      'taxonomy_term',
      'default',
      ['target_bundles' => ['tags']],
    );

    // Now install the test module that provides the View config depending
    // on taxonomy.vocabulary.tags and node__field_tags.
    $this->container->get('module_installer')->install(['protect_views_flood_control_test']);
    $this->rebuildContainer();
  }

  /**
   * Test that a 429 response is returned.
   *
   * Flood threshold = 2 in the provided View YAML; window = 30 seconds.
   * Third non-AJAX submission should return HTTP 429 with Retry-After: 30.
   */
  public function testFloodControl429(): void {
    // Create sample content for the view.
    $term1 = $this->createTerm($this->vocab, ['name' => 'tag1']);
    $term2 = $this->createTerm($this->vocab, ['name' => 'tag2']);
    $term3 = $this->createTerm($this->vocab, ['name' => 'tag3']);
    $this->drupalCreateNode([
      'type' => 'article',
      'title' => 'test node 1',
      'field_tags' => [['target_id' => $term1->id()]],
    ]);
    $this->drupalCreateNode([
      'type' => 'article',
      'title' => 'test node 2',
      'field_tags' => [['target_id' => $term2->id()]],
    ]);
    $this->drupalCreateNode([
      'type' => 'article',
      'title' => 'test node 3',
      'field_tags' => [['target_id' => $term3->id()]],
    ]);

    // Load the page display shipped by the test module.
    $this->drupalGet('test-flood-view');
    $this->assertSession()->statusCodeEquals(200);

    // 1st submission (allowed).
    $this->submitForm([
      'field_tags_target_id' => (string) $term1->id(),
    ], 'Apply');
    $this->assertSession()->statusCodeEquals(200);

    // 2nd submission (allowed).
    $this->submitForm([
      'field_tags_target_id' => (string) $term2->id(),
    ], 'Apply');
    $this->assertSession()->statusCodeEquals(200);

    // 3rd submission but to an already cached result (allowed).
    $this->submitForm([
      'field_tags_target_id' => (string) $term1->id(),
    ], 'Apply');
    $this->assertSession()->statusCodeEquals(200);

    // 3rd submission within the 30s window -> should be blocked with 429.
    $this->submitForm([
      'field_tags_target_id' => (string) $term3->id(),
    ], 'Apply');

    $this->assertSession()->statusCodeEquals(429);
    $this->assertSession()->responseHeaderContains('Retry-After', '30');
    $this->assertSession()->pageTextContains('Client error');
  }

}
