# Public APIs Integration - Usage Examples

This document provides practical examples of how to use the Public APIs Integration module.

## REST API Examples

### Using cURL

#### Get all available categories
```bash
curl -X GET "https://yoursite.com/api/public-apis/categories" \
  -H "Accept: application/json"
```

#### Get APIs in the Animals category
```bash
curl -X GET "https://yoursite.com/api/public-apis/animals" \
  -H "Accept: application/json"
```

#### Get a random cat fact
```bash
curl -X GET "https://yoursite.com/api/public-apis/animals/cat_facts?endpoint=fact" \
  -H "Accept: application/json"
```

#### Get weather data (requires Weatherstack API key)
```bash
curl -X GET "https://yoursite.com/api/public-apis/weather/weatherstack?endpoint=current&query=London" \
  -H "Accept: application/json"
```

#### Search for APIs containing "weather"
```bash
curl -X GET "https://yoursite.com/api/public-apis/search?q=weather" \
  -H "Accept: application/json"
```

### Using JavaScript/jQuery

```javascript
// Get all categories
$.ajax({
  url: '/api/public-apis/categories',
  method: 'GET',
  dataType: 'json',
  success: function(data) {
    console.log('Available categories:', data.data);
  }
});

// Get a random dog image
$.ajax({
  url: '/api/public-apis/animals/random_dog',
  method: 'GET',
  data: { endpoint: 'woof.json' },
  dataType: 'json',
  success: function(data) {
    if (data.success) {
      console.log('Dog image URL:', data.data.url);
    }
  }
});

// Search for finance APIs
$.ajax({
  url: '/api/public-apis/search',
  method: 'GET',
  data: { q: 'finance' },
  dataType: 'json',
  success: function(data) {
    console.log('Finance APIs:', data.data.results);
  }
});
```

### Using PHP

```php
// Example: Get cat facts using Drupal's HTTP client
$client = \Drupal::httpClient();
$api_url = 'https://yoursite.com/api/public-apis/animals/cat_facts';

try {
  $response = $client->get($api_url, [
    'query' => ['endpoint' => 'fact'],
    'headers' => ['Accept' => 'application/json'],
  ]);
  
  $data = json_decode($response->getBody(), TRUE);
  
  if ($data['success']) {
    $cat_fact = $data['data']['fact'];
    \Drupal::messenger()->addMessage("Cat fact: " . $cat_fact);
  }
} catch (Exception $e) {
  \Drupal::logger('my_module')->error('API call failed: @error', [
    '@error' => $e->getMessage(),
  ]);
}
```

## Drupal Service Usage

### Injecting the API Client Service

```php
<?php

namespace Drupal\my_module\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\public_apis\Service\PublicApisClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

class MyController extends ControllerBase {

  protected $apiClient;

  public function __construct(PublicApisClient $api_client) {
    $this->apiClient = $api_client;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('public_apis.api_client')
    );
  }

  public function getRandomFact() {
    $result = $this->apiClient->makeApiCall('animals', 'cat_facts', 'fact');
    
    if ($result['success']) {
      return [
        '#markup' => '<p>' . $result['data']['fact'] . '</p>',
      ];
    } else {
      return [
        '#markup' => '<p>Error: ' . $result['error'] . '</p>',
      ];
    }
  }
}
```

### Custom API Integration

```php
<?php

namespace Drupal\my_module\Service;

use Drupal\public_apis\Service\PublicApisClient;

class MyCustomApiService {

  protected $apiClient;

  public function __construct(PublicApisClient $api_client) {
    $this->apiClient = $api_client;
  }

  public function getWeatherForCity($city) {
    $result = $this->apiClient->makeApiCall(
      'weather', 
      'weatherstack', 
      'current', 
      ['query' => $city]
    );

    if ($result['success']) {
      return [
        'temperature' => $result['data']['current']['temperature'],
        'description' => $result['data']['current']['weather_descriptions'][0],
        'city' => $result['data']['location']['name'],
      ];
    }

    return NULL;
  }

  public function getCurrencyRate($from, $to) {
    $result = $this->apiClient->makeApiCall(
      'finance',
      'fixer',
      'latest',
      [
        'base' => $from,
        'symbols' => $to,
      ]
    );

    if ($result['success']) {
      return $result['data']['rates'][$to];
    }

    return NULL;
  }
}
```

## Block Plugin Example

```php
<?php

namespace Drupal\my_module\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\public_apis\Service\PublicApisClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * @Block(
 *   id = "cat_fact_block",
 *   admin_label = @Translation("Random Cat Fact"),
 * )
 */
class CatFactBlock extends BlockBase implements ContainerFactoryPluginInterface {

  protected $apiClient;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, PublicApisClient $api_client) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->apiClient = $api_client;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('public_apis.api_client')
    );
  }

  public function build() {
    $result = $this->apiClient->makeApiCall('animals', 'cat_facts', 'fact');

    if ($result['success']) {
      return [
        '#theme' => 'item_list',
        '#title' => $this->t('Random Cat Fact'),
        '#items' => [$result['data']['fact']],
        '#cache' => [
          'max-age' => 300, // Cache for 5 minutes
        ],
      ];
    }

    return [
      '#markup' => $this->t('Unable to load cat fact at this time.'),
    ];
  }
}
```

## Twig Template Usage

```twig
{# In a custom template #}
{% set weather_data = api_call('weather', 'weatherstack', 'current', {'query': 'London'}) %}

{% if weather_data.success %}
  <div class="weather-widget">
    <h3>Weather in {{ weather_data.data.location.name }}</h3>
    <p>Temperature: {{ weather_data.data.current.temperature }}°C</p>
    <p>{{ weather_data.data.current.weather_descriptions.0 }}</p>
  </div>
{% else %}
  <p>Weather data unavailable</p>
{% endif %}
```

## AJAX Example

```javascript
// Example AJAX form for API testing
(function ($, Drupal) {
  'use strict';

  Drupal.behaviors.apiTester = {
    attach: function (context, settings) {
      $('#api-test-form', context).once('api-tester').on('submit', function (e) {
        e.preventDefault();
        
        var category = $('#category').val();
        var apiName = $('#api-name').val();
        var endpoint = $('#endpoint').val();
        
        var url = '/api/public-apis/' + category + '/' + apiName;
        
        $.ajax({
          url: url,
          method: 'GET',
          data: { endpoint: endpoint },
          dataType: 'json',
          success: function (data) {
            $('#api-results').html('<pre>' + JSON.stringify(data, null, 2) + '</pre>');
          },
          error: function (xhr, status, error) {
            $('#api-results').html('<div class="error">Error: ' + error + '</div>');
          }
        });
      });
    }
  };

})(jQuery, Drupal);
```

## Response Caching

The module automatically caches GET requests. You can check cache status:

```php
// Check if a response is cached
$cache_key = "public_apis:animals:cat_facts:" . md5('fact' . serialize([]));
$cached = \Drupal::cache()->get($cache_key);

if ($cached) {
  // Use cached data
  $data = $cached->data;
} else {
  // Make fresh API call
  $result = $api_client->makeApiCall('animals', 'cat_facts', 'fact');
}
```

## Rate Limiting

Check rate limit status:

```php
$rate_limiter = \Drupal::service('public_apis.rate_limiter');
$status = $rate_limiter->getRateLimitStatus('cat_facts');

// Returns:
// [
//   'minute' => ['current' => 5, 'limit' => 60, 'remaining' => 55, 'reset_time' => 1234567890],
//   'hour' => [...],
//   'day' => [...]
// ]
```

## Error Handling

```php
$result = $api_client->makeApiCall('animals', 'cat_facts', 'fact');

switch ($result['status_code']) {
  case 200:
    // Success
    break;
  case 429:
    // Rate limited
    break;
  case 401:
    // Authentication required
    break;
  case 404:
    // API not found
    break;
  default:
    // Other error
    break;
}
```

These examples demonstrate the flexibility and power of the Public APIs Integration module for Drupal 11.