# Public APIs Integration Module

A comprehensive Drupal 11 module that provides a unified interface for accessing various public APIs as listed in the [public-apis repository](https://github.com/public-apis/public-apis).

## Features

- **Multi-Category API Support**: Access APIs from Animals, Weather, Finance, Anime, Books, and many more categories
- **Rate Limiting**: Configurable rate limiting to prevent API abuse
- **Caching**: Built-in caching system for improved performance
- **Authentication Management**: Support for API keys, Bearer tokens, and Basic authentication
- **Security Features**: IP restrictions, HTTPS enforcement, and security headers
- **RESTful Endpoints**: Clean REST API endpoints for all integrated services
- **Administrative Dashboard**: Monitor and manage API integrations
- **Drupal 11 Compatible**: Built using Drupal 11 best practices and APIs

## Installation

1. Extract the module to your Drupal `modules/custom` directory
2. Enable the module through the admin interface or using Drush:
   ```bash
   drush en public_apis
   ```
3. Configure the module at `/admin/config/services/public-apis`

## Configuration

### Basic Settings

Navigate to **Administration > Configuration > Web Services > Public APIs** to configure:

- **Request Timeout**: Maximum time to wait for API responses (default: 30 seconds)
- **Cache TTL**: How long to cache API responses (default: 1 hour)
- **Rate Limiting**: Enable/disable rate limiting with configurable limits

### API Keys

Configure API keys for services that require authentication:

- **Weatherstack**: For weather data
- **Fixer**: For currency exchange rates
- **Various other APIs**: As needed based on your usage

### Security Settings

- **IP Restrictions**: Limit API access to specific IP addresses
- **HTTPS Enforcement**: Require HTTPS for all API endpoints
- **Logging**: Configure logging for API calls and errors

## Usage

### Available Endpoints

#### List All Categories
```
GET /api/public-apis/categories
```

#### List APIs in a Category
```
GET /api/public-apis/{category}
```

#### Make API Call
```
GET|POST /api/public-apis/{category}/{api_name}?endpoint={endpoint}&param1=value1
```

#### Search APIs
```
GET /api/public-apis/search?q={search_term}
```

### Examples

#### Get Cat Facts
```bash
curl -X GET "https://yoursite.com/api/public-apis/animals/cat_facts?endpoint=fact"
```

#### Get Weather Data (requires API key)
```bash
curl -X GET "https://yoursite.com/api/public-apis/weather/weatherstack?endpoint=current&query=London"
```

#### Search for APIs
```bash
curl -X GET "https://yoursite.com/api/public-apis/search?q=weather"
```

### Response Format

All endpoints return JSON responses in the following format:

```json
{
  "success": true,
  "data": {
    // API response data
  },
  "api_info": {
    "category": "animals",
    "api_name": "cat_facts",
    "endpoint": "fact",
    "method": "GET"
  },
  "meta": {
    "status_code": 200,
    "timestamp": 1234567890
  }
}
```

### Error Responses

```json
{
  "success": false,
  "error": "Rate limit exceeded",
  "api_info": {
    "category": "animals",
    "api_name": "cat_facts",
    "endpoint": "fact",
    "method": "GET"
  }
}
```

## Supported API Categories

### Animals
- Cat Facts
- Dog Facts
- Random Dog Images
- HTTP Cat/Dog
- And more...

### Weather
- Weatherstack (requires API key)

### Finance
- Fixer (currency exchange rates, requires API key)

### Anime
- Jikan (MyAnimeList API)
- Studio Ghibli API

### Books
- Open Library

### Art & Design
- Art Institute of Chicago

### Test Data
- JSONPlaceholder

## Permissions

The module defines two permissions:

- **Use Public APIs**: Allows access to API endpoints
- **Administer Public APIs**: Allows configuration and dashboard access

## Rate Limiting

Default rate limits:
- 60 requests per minute
- 1,000 requests per hour
- 10,000 requests per day

Rate limits are configurable per user/IP and can be adjusted in the admin interface.

## Caching

API responses are cached to improve performance:
- GET requests are cached by default
- Cache TTL is configurable (default: 1 hour)
- Cache keys include endpoint and parameters

## Security Features

- **IP Restrictions**: Whitelist specific IP addresses
- **HTTPS Enforcement**: Require secure connections
- **Security Headers**: X-Content-Type-Options, X-Frame-Options, X-XSS-Protection
- **Input Validation**: All user inputs are validated and sanitized
- **Permission Checks**: All endpoints require appropriate permissions

## Extending the Module

### Adding New APIs

To add new APIs, update the `ApiRegistry` service:

```php
// In src/Service/ApiRegistry.php
'new_category' => [
  'new_api' => [
    'name' => 'New API',
    'base_url' => 'https://api.example.com',
    'endpoints' => ['endpoint1', 'endpoint2'],
    'auth_type' => 'apiKey', // or 'none', 'bearer', 'basic'
    'auth_config' => [
      'location' => 'header', // or 'query'
      'name' => 'X-API-Key',
    ],
    'https' => true,
    'cors' => true,
    'description' => 'Description of the API',
  ],
],
```

### Custom Controllers

Extend the `PublicApisController` class to add custom endpoints:

```php
<?php

namespace Drupal\my_module\Controller;

use Drupal\public_apis\Controller\PublicApisController;

class MyCustomController extends PublicApisController {
  
  public function customEndpoint() {
    // Your custom logic here
  }
}
```

## Troubleshooting

### Common Issues

1. **403 Forbidden**: Check permissions and IP restrictions
2. **Rate Limit Exceeded**: Adjust rate limits or wait for reset
3. **API Key Invalid**: Verify API keys in configuration
4. **Timeout Errors**: Increase request timeout setting

### Debugging

Enable logging to troubleshoot issues:
1. Go to **Administration > Configuration > Web Services > Public APIs**
2. Enable "Log API Calls" and "Log Failed Calls"
3. Check logs at **Administration > Reports > Recent log messages**

### Log Entries

The module logs:
- Successful API calls (if enabled)
- Failed API calls
- Rate limit violations
- Security violations

## Performance Optimization

1. **Enable Caching**: Use appropriate cache TTL values
2. **Rate Limiting**: Prevent abuse and maintain performance
3. **HTTP Client**: Uses Guzzle for efficient HTTP requests
4. **Database**: Minimal database usage, relies on caching

## Development

### Running Tests

```bash
# PHPUnit tests
./vendor/bin/phpunit modules/custom/public_apis/tests/

# Drupal test runner
php core/scripts/run-tests.sh --module public_apis
```

### Code Standards

The module follows Drupal coding standards:
- PSR-4 autoloading
- Drupal coding standards
- Type declarations
- Proper documentation

## Contributing

1. Fork the repository
2. Create a feature branch
3. Follow Drupal coding standards
4. Add tests for new functionality
5. Submit a pull request

## License

This module is licensed under the GPL-2.0+ license, compatible with Drupal core.

## Support

- Create issues for bugs or feature requests
- Check the Drupal.org project page for updates
- Join the community discussions

## Changelog

### Version 1.0.0
- Initial release
- Support for major API categories
- Rate limiting and caching
- Security features
- Administrative interface
- Drupal 11 compatibility