/**
 * @file
 * API Explorer JavaScript for Public APIs Integration module.
 */

(function ($, Drupal, drupalSettings, once) {
  'use strict';

  /**
   * API Explorer functionality.
   */
  Drupal.behaviors.publicApisExplorer = {
    attach: function (context, settings) {
      once('public-apis-explorer', '.public-apis-explorer', context).forEach(function (element) {
        var $explorer = $(element);
        var apiData = drupalSettings.publicApis ? drupalSettings.publicApis.apiData || {} : {};
        var baseUrl = drupalSettings.publicApis ? drupalSettings.publicApis.baseUrl || '' : '';
        var currentApi = null;

        // Initialize explorer
        initExplorer();

        function initExplorer() {
          bindEvents();
          setupFormValidation();
        }

        function bindEvents() {
          // Category toggle functionality
          $explorer.on('click', '.category-toggle', function(e) {
            e.preventDefault();
            var $button = $(this);
            var $list = $button.closest('.category-group').find('.api-list');
            var isExpanded = $button.attr('aria-expanded') === 'true';
            
            if (isExpanded) {
              $list.slideUp(200);
              $button.attr('aria-expanded', 'false');
              $button.find('.toggle-icon').text('▶');
            } else {
              $list.slideDown(200);
              $button.attr('aria-expanded', 'true');
              $button.find('.toggle-icon').text('▼');
            }
          });

          // API selection
          $explorer.on('click', '.api-item', function() {
            var $item = $(this);
            var category = $item.data('category');
            var api = $item.data('api');
            
            // Update active state
            $('.api-item').removeClass('active');
            $item.addClass('active');
            
            // Load API configuration
            loadApiConfig(category, api);
          });

          // Send request button
          $explorer.on('click', '#send-request', function() {
            if (currentApi) {
              sendApiRequest();
            }
          });

          // Clear form button
          $explorer.on('click', '#clear-form', function() {
            clearForm();
          });

          // Response tabs
          $explorer.on('click', '.tab-button', function() {
            var $button = $(this);
            var tab = $button.data('tab');
            
            $('.tab-button').removeClass('active');
            $button.addClass('active');
            
            $('.tab-content').removeClass('active');
            $('#' + tab + '-response, #response-' + tab).addClass('active');
          });

          // Auto-format JSON in textareas
          $explorer.on('blur', '#api-params, #api-headers', function() {
            var $textarea = $(this);
            var content = $textarea.val().trim();
            
            if (content) {
              try {
                var parsed = JSON.parse(content);
                var formatted = JSON.stringify(parsed, null, 2);
                $textarea.val(formatted);
                $textarea.removeClass('error');
              } catch (e) {
                $textarea.addClass('error');
              }
            }
          });
        }

        function setupFormValidation() {
          // Add real-time validation for JSON fields
          $('#api-params, #api-headers').on('input', function() {
            var $textarea = $(this);
            var content = $textarea.val().trim();
            
            if (content) {
              try {
                JSON.parse(content);
                $textarea.removeClass('error');
              } catch (e) {
                $textarea.addClass('error');
              }
            } else {
              $textarea.removeClass('error');
            }
          });
        }

        function loadApiConfig(category, api) {
          if (!apiData[category] || !apiData[category].apis[api]) {
            showError('API configuration not found');
            return;
          }

          currentApi = {
            category: category,
            api: api,
            config: apiData[category].apis[api]
          };

          var config = currentApi.config;
          
          // Update panel header
          $('.api-title').text(config.name || api);
          $('.api-category').text(apiData[category].name);
          $('.api-description').text(config.description || 'No description available');
          
          // Populate endpoints
          var $endpointSelect = $('#api-endpoint');
          $endpointSelect.empty().append('<option value="">Select an endpoint</option>');
          
          if (config.endpoints && config.endpoints.length > 0) {
            config.endpoints.forEach(function(endpoint) {
              var displayName = endpoint.path || endpoint.name || endpoint;
              var value = endpoint.path || endpoint;
              $endpointSelect.append('<option value="' + value + '">' + displayName + '</option>');
            });
          } else {
            $endpointSelect.append('<option value="/">/</option>');
          }
          
          // Show the config panel and hide welcome message
          $('.welcome-message').hide();
          $('.api-config-panel').show();
          
          // Reset form
          clearForm();
        }

        function sendApiRequest() {
          if (!currentApi) {
            showError('No API selected');
            return;
          }

          var endpoint = $('#api-endpoint').val() || '/';
          var method = $('#api-method').val() || 'GET';
          var params = parseJsonField('#api-params');
          var headers = parseJsonField('#api-headers');

          if (params === null || headers === null) {
            showError('Invalid JSON in parameters or headers');
            return;
          }

          // Show loading
          showLoading(true);
          
          var requestData = {
            endpoint: endpoint,
            method: method
          };

          // Add parameters based on method
          if (method === 'GET') {
            // For GET requests, add params to URL
            if (params && Object.keys(params).length > 0) {
              var queryString = $.param(params);
              requestData.endpoint = endpoint + (endpoint.includes('?') ? '&' : '?') + queryString;
            }
          } else {
            // For other methods, send as request body
            if (params && Object.keys(params).length > 0) {
              requestData.body = JSON.stringify(params);
            }
          }

          var startTime = Date.now();
          
          // Make the request
          $.ajax({
            url: baseUrl + '/api/public-apis/' + currentApi.category + '/' + currentApi.api,
            method: method,
            data: method === 'GET' ? (params || {}) : JSON.stringify(params || {}),
            contentType: method === 'GET' ? undefined : 'application/json',
            headers: headers || {},
            timeout: 30000
          })
          .done(function(data, textStatus, xhr) {
            var responseTime = Date.now() - startTime;
            displayResponse(xhr, data, responseTime, true);
          })
          .fail(function(xhr, textStatus, errorThrown) {
            var responseTime = Date.now() - startTime;
            var errorData = {
              error: textStatus === 'timeout' ? 'Request timeout' : errorThrown,
              message: xhr.responseText || 'Request failed'
            };
            
            try {
              errorData = JSON.parse(xhr.responseText);
            } catch (e) {
              // Use default error data
            }
            
            displayResponse(xhr, errorData, responseTime, false);
          })
          .always(function() {
            showLoading(false);
          });
        }

        function parseJsonField(selector) {
          var content = $(selector).val().trim();
          if (!content) return {};
          
          try {
            return JSON.parse(content);
          } catch (e) {
            return null;
          }
        }

        function displayResponse(xhr, data, responseTime, success) {
          var $responseSection = $('.response-section');
          
          // Update meta information
          $('.status-code').text(xhr.status + ' ' + xhr.statusText)
            .removeClass('success error')
            .addClass(success ? 'success' : 'error');
          $('.response-time').text(responseTime + 'ms');
          $('.response-size').text(formatBytes(JSON.stringify(data).length));
          
          // Format and display response data
          var formattedJson = JSON.stringify(data, null, 2);
          $('.formatted-json').text(formattedJson);
          $('.raw-json').text(JSON.stringify(data));
          
          // Display response headers
          var headersText = '';
          if (xhr.getAllResponseHeaders) {
            headersText = xhr.getAllResponseHeaders();
          }
          $('.headers-display').text(headersText);
          
          // Show response section
          $responseSection.show();
          
          // Scroll to response
          $responseSection[0].scrollIntoView({ behavior: 'smooth' });
        }

        function clearForm() {
          $('#api-endpoint').val('');
          $('#api-method').val('GET');
          $('#api-params').val('').removeClass('error');
          $('#api-headers').val('').removeClass('error');
          $('.response-section').hide();
        }

        function showLoading(show) {
          var $overlay = $('.loading-overlay');
          if (show) {
            $overlay.show();
          } else {
            $overlay.hide();
          }
        }

        function showError(message) {
          // Create a simple error notification
          var $error = $('<div class="api-error">' + message + '</div>');
          $explorer.prepend($error);
          
          setTimeout(function() {
            $error.fadeOut(function() {
              $error.remove();
            });
          }, 3000);
        }

        function formatBytes(bytes) {
          if (bytes === 0) return '0 Bytes';
          var k = 1024;
          var sizes = ['Bytes', 'KB', 'MB', 'GB'];
          var i = Math.floor(Math.log(bytes) / Math.log(k));
          return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
      });
    }
  };

})(jQuery, Drupal, drupalSettings, once);