<?php

namespace Drupal\public_apis\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\public_apis\Service\PublicApisClient;
use Drupal\public_apis\Service\ApiRegistry;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for Public APIs integration endpoints.
 */
class PublicApisController extends ControllerBase {

  /**
   * The public APIs client service.
   *
   * @var \Drupal\public_apis\Service\PublicApisClient
   */
  protected $apiClient;

  /**
   * The API registry service.
   *
   * @var \Drupal\public_apis\Service\ApiRegistry
   */
  protected $apiRegistry;

  /**
   * Constructs a new PublicApisController object.
   *
   * @param \Drupal\public_apis\Service\PublicApisClient $api_client
   *   The public APIs client service.
   * @param \Drupal\public_apis\Service\ApiRegistry $api_registry
   *   The API registry service.
   */
  public function __construct(PublicApisClient $api_client, ApiRegistry $api_registry) {
    $this->apiClient = $api_client;
    $this->apiRegistry = $api_registry;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('public_apis.api_client'),
      $container->get('public_apis.api_registry')
    );
  }

  /**
   * Makes an API call to a specified public API.
   *
   * @param string $category
   *   The API category.
   * @param string $api_name
   *   The API name.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with API data or error.
   */
  public function apiCall(string $category, string $api_name, Request $request): JsonResponse {
    try {
      // Validate API exists.
      $api_config = $this->apiRegistry->getApiConfig($category, $api_name);
      if (!$api_config) {
        return new JsonResponse([
          'error' => 'API not found',
          'message' => "API '{$api_name}' not found in category '{$category}'",
        ], Response::HTTP_NOT_FOUND);
      }

      // Get endpoint and parameters from request.
      $endpoint = $request->query->get('endpoint', '');
      $method = $request->getMethod();
      
      // Get parameters based on method.
      if ($method === 'GET') {
        $parameters = $request->query->all();
        // Remove our internal parameters.
        unset($parameters['endpoint']);
      } else {
        $content = $request->getContent();
        $parameters = $content ? json_decode($content, TRUE) ?: [] : [];
      }

      // Make the API call.
      $result = $this->apiClient->makeApiCall($category, $api_name, $endpoint, $parameters, $method);

      if ($result['success']) {
        return new JsonResponse([
          'success' => TRUE,
          'data' => $result['data'],
          'api_info' => [
            'category' => $category,
            'api_name' => $api_name,
            'endpoint' => $endpoint,
            'method' => $method,
          ],
          'meta' => [
            'status_code' => $result['status_code'],
            'timestamp' => time(),
          ],
        ]);
      } else {
        $status_code = $result['status_code'] ?: Response::HTTP_INTERNAL_SERVER_ERROR;
        return new JsonResponse([
          'success' => FALSE,
          'error' => $result['error'],
          'api_info' => [
            'category' => $category,
            'api_name' => $api_name,
            'endpoint' => $endpoint,
            'method' => $method,
          ],
        ], $status_code);
      }

    } catch (\Exception $e) {
      $this->getLogger('public_apis')->error('Controller error: @error', [
        '@error' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => 'Internal server error',
        'message' => $e->getMessage(),
      ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
  }

  /**
   * Returns all available API categories.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with categories data.
   */
  public function getCategories(): JsonResponse {
    try {
      $categories = $this->apiRegistry->getCategories();

      return new JsonResponse([
        'success' => TRUE,
        'data' => $categories,
        'meta' => [
          'total_categories' => count($categories),
          'timestamp' => time(),
        ],
      ]);

    } catch (\Exception $e) {
      $this->getLogger('public_apis')->error('Error getting categories: @error', [
        '@error' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => 'Failed to retrieve categories',
      ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
  }

  /**
   * Returns APIs for a specific category.
   *
   * @param string $category
   *   The category name.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with APIs data.
   */
  public function getApiList(string $category): JsonResponse {
    try {
      $apis = $this->apiRegistry->getApisByCategory($category);

      if (empty($apis)) {
        return new JsonResponse([
          'success' => FALSE,
          'error' => 'Category not found',
          'message' => "Category '{$category}' not found or contains no APIs",
        ], Response::HTTP_NOT_FOUND);
      }

      return new JsonResponse([
        'success' => TRUE,
        'data' => [
          'category' => $category,
          'apis' => $apis,
        ],
        'meta' => [
          'total_apis' => count($apis),
          'timestamp' => time(),
        ],
      ]);

    } catch (\Exception $e) {
      $this->getLogger('public_apis')->error('Error getting API list: @error', [
        '@error' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => 'Failed to retrieve API list',
      ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
  }

  /**
   * Displays the Public APIs dashboard.
   *
   * @return array
   *   Render array for the dashboard page.
   */
  public function dashboard(): array {
    $categories = $this->apiRegistry->getCategories();
    $total_apis = array_sum(array_column($categories, 'count'));

    $build = [
      '#theme' => 'public_apis_dashboard',
      '#categories' => $categories,
      '#total_apis' => $total_apis,
      '#attached' => [
        'library' => [
          'public_apis/dashboard',
        ],
      ],
    ];

    return $build;
  }

  /**
   * Access callback for API endpoints.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'use public apis');
  }

  /**
   * Access callback for admin endpoints.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function adminAccess(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'administer public apis');
  }

  /**
   * API endpoint for searching APIs.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with search results.
   */
  public function searchApis(Request $request): JsonResponse {
    $query = $request->query->get('q', '');

    if (empty($query)) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => 'Search query is required',
      ], Response::HTTP_BAD_REQUEST);
    }

    try {
      $results = $this->apiRegistry->searchApis($query);

      return new JsonResponse([
        'success' => TRUE,
        'data' => [
          'query' => $query,
          'results' => $results,
        ],
        'meta' => [
          'total_results' => count($results),
          'timestamp' => time(),
        ],
      ]);

    } catch (\Exception $e) {
      $this->getLogger('public_apis')->error('Error searching APIs: @error', [
        '@error' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => 'Search failed',
      ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
  }

  /**
   * Display the interactive API explorer interface.
   *
   * @return array
   *   Render array for the API explorer page.
   */
  public function apiExplorer(): array {
    // Get all available APIs for the explorer
    $categories = $this->apiRegistry->getCategories();
    
    // Prepare API data for JavaScript
    $api_data = [];
    foreach ($categories as $category_slug => $category) {
      $api_data[$category_slug] = [
        'name' => $category['name'],
        'apis' => []
      ];
      
      foreach ($category['apis'] as $api_slug) {
        $api_config = $this->apiRegistry->getApiConfig($category_slug, $api_slug);
        if ($api_config) {
          $api_data[$category_slug]['apis'][$api_slug] = [
            'name' => $api_config['name'] ?? $api_slug,
            'description' => $api_config['description'] ?? '',
            'base_url' => $api_config['base_url'] ?? '',
            'endpoints' => $api_config['endpoints'] ?? [],
            'auth_required' => $api_config['auth_required'] ?? FALSE,
          ];
        }
      }
    }

    return [
      '#theme' => 'public_apis_explorer',
      '#categories' => $categories,
      '#api_data' => $api_data,
      '#attached' => [
        'library' => [
          'public_apis/explorer',
        ],
        'drupalSettings' => [
          'publicApis' => [
            'apiData' => $api_data,
            'baseUrl' => \Drupal::request()->getSchemeAndHttpHost(),
          ],
        ],
      ],
    ];
  }

}