<?php

namespace Drupal\public_apis\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpFoundation\Request;

/**
 * Event subscriber for API security checks.
 */
class ApiSecuritySubscriber implements EventSubscriberInterface {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Constructs an ApiSecuritySubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   */
  public function __construct(ConfigFactoryInterface $config_factory, AccountInterface $current_user) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events = [];
    $events[KernelEvents::REQUEST][] = ['onRequest', 100];
    return $events;
  }

  /**
   * Handles the request event.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onRequest(RequestEvent $event): void {
    $request = $event->getRequest();

    // Only check API routes.
    $route_name = $request->attributes->get('_route');
    if (!$route_name || strpos($route_name, 'public_apis.api_') !== 0) {
      return;
    }

    $config = $this->configFactory->get('public_apis.settings');

    // Check HTTPS requirement.
    if ($config->get('require_https') && !$request->isSecure()) {
      throw new AccessDeniedHttpException('HTTPS is required for API access.');
    }

    // Check IP restrictions.
    $allowed_ips = $config->get('allowed_ips');
    if (!empty($allowed_ips)) {
      $ips = array_filter(array_map('trim', explode("\n", $allowed_ips)));
      $client_ip = $request->getClientIp();
      
      if (!in_array($client_ip, $ips)) {
        throw new AccessDeniedHttpException('Access denied from this IP address.');
      }
    }

    // Additional security headers.
    $response = $event->getResponse();
    if ($response) {
      $response->headers->set('X-Content-Type-Options', 'nosniff');
      $response->headers->set('X-Frame-Options', 'DENY');
      $response->headers->set('X-XSS-Protection', '1; mode=block');
    }
  }

}