<?php

namespace Drupal\public_apis\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Component\Datetime\TimeInterface;

/**
 * Service for implementing rate limiting for API calls.
 */
class RateLimiter {

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Default rate limits per API.
   *
   * @var array
   */
  protected $defaultLimits = [
    'requests_per_minute' => 60,
    'requests_per_hour' => 1000,
    'requests_per_day' => 10000,
  ];

  /**
   * Constructs a RateLimiter object.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(CacheBackendInterface $cache, TimeInterface $time) {
    $this->cache = $cache;
    $this->time = $time;
  }

  /**
   * Checks if a request is allowed based on rate limits.
   *
   * @param string $api_name
   *   The API name.
   * @param string $identifier
   *   Optional identifier (e.g., user ID, IP address).
   *
   * @return bool
   *   TRUE if the request is allowed, FALSE otherwise.
   */
  public function allowRequest(string $api_name, string $identifier = 'global'): bool {
    $now = $this->time->getCurrentTime();

    // Check minute limit.
    if (!$this->checkLimit($api_name, $identifier, 'minute', $now, $this->defaultLimits['requests_per_minute'])) {
      return FALSE;
    }

    // Check hour limit.
    if (!$this->checkLimit($api_name, $identifier, 'hour', $now, $this->defaultLimits['requests_per_hour'])) {
      return FALSE;
    }

    // Check day limit.
    if (!$this->checkLimit($api_name, $identifier, 'day', $now, $this->defaultLimits['requests_per_day'])) {
      return FALSE;
    }

    // All checks passed, increment counters.
    $this->incrementCounter($api_name, $identifier, 'minute', $now);
    $this->incrementCounter($api_name, $identifier, 'hour', $now);
    $this->incrementCounter($api_name, $identifier, 'day', $now);

    return TRUE;
  }

  /**
   * Checks if a request is within the rate limit for a specific time window.
   *
   * @param string $api_name
   *   The API name.
   * @param string $identifier
   *   The identifier.
   * @param string $window
   *   The time window (minute, hour, day).
   * @param int $now
   *   Current timestamp.
   * @param int $limit
   *   The rate limit for this window.
   *
   * @return bool
   *   TRUE if within limit, FALSE otherwise.
   */
  protected function checkLimit(string $api_name, string $identifier, string $window, int $now, int $limit): bool {
    $window_start = $this->getWindowStart($now, $window);
    $cache_key = "rate_limit:{$api_name}:{$identifier}:{$window}:{$window_start}";

    $cached = $this->cache->get($cache_key);
    $current_count = $cached ? $cached->data : 0;

    return $current_count < $limit;
  }

  /**
   * Increments the request counter for a specific time window.
   *
   * @param string $api_name
   *   The API name.
   * @param string $identifier
   *   The identifier.
   * @param string $window
   *   The time window.
   * @param int $now
   *   Current timestamp.
   */
  protected function incrementCounter(string $api_name, string $identifier, string $window, int $now): void {
    $window_start = $this->getWindowStart($now, $window);
    $cache_key = "rate_limit:{$api_name}:{$identifier}:{$window}:{$window_start}";

    $cached = $this->cache->get($cache_key);
    $current_count = $cached ? $cached->data : 0;
    $new_count = $current_count + 1;

    $expire = $window_start + $this->getWindowDuration($window);
    $this->cache->set($cache_key, $new_count, $expire);
  }

  /**
   * Gets the start of the current time window.
   *
   * @param int $timestamp
   *   The timestamp.
   * @param string $window
   *   The time window.
   *
   * @return int
   *   The window start timestamp.
   */
  protected function getWindowStart(int $timestamp, string $window): int {
    switch ($window) {
      case 'minute':
        return floor($timestamp / 60) * 60;

      case 'hour':
        return floor($timestamp / 3600) * 3600;

      case 'day':
        return strtotime('today', $timestamp);

      default:
        return $timestamp;
    }
  }

  /**
   * Gets the duration of a time window in seconds.
   *
   * @param string $window
   *   The time window.
   *
   * @return int
   *   The duration in seconds.
   */
  protected function getWindowDuration(string $window): int {
    switch ($window) {
      case 'minute':
        return 60;

      case 'hour':
        return 3600;

      case 'day':
        return 86400;

      default:
        return 3600;
    }
  }

  /**
   * Gets the current rate limit status for an API.
   *
   * @param string $api_name
   *   The API name.
   * @param string $identifier
   *   The identifier.
   *
   * @return array
   *   Rate limit status information.
   */
  public function getRateLimitStatus(string $api_name, string $identifier = 'global'): array {
    $now = $this->time->getCurrentTime();
    $status = [];

    foreach (['minute', 'hour', 'day'] as $window) {
      $window_start = $this->getWindowStart($now, $window);
      $cache_key = "rate_limit:{$api_name}:{$identifier}:{$window}:{$window_start}";

      $cached = $this->cache->get($cache_key);
      $current_count = $cached ? $cached->data : 0;

      $limit_key = "requests_per_{$window}";
      $limit = $this->defaultLimits[$limit_key];

      $status[$window] = [
        'current' => $current_count,
        'limit' => $limit,
        'remaining' => max(0, $limit - $current_count),
        'reset_time' => $window_start + $this->getWindowDuration($window),
      ];
    }

    return $status;
  }

}