<?php

namespace Drupal\public_apis\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\public_apis\Service\ApiRegistry;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Public APIs integration settings.
 */
class PublicApisSettingsForm extends ConfigFormBase {

  /**
   * The API registry service.
   *
   * @var \Drupal\public_apis\Service\ApiRegistry
   */
  protected $apiRegistry;

  /**
   * Constructs a PublicApisSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\public_apis\Service\ApiRegistry $api_registry
   *   The API registry service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, ApiRegistry $api_registry) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->apiRegistry = $api_registry;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('public_apis.api_registry')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['public_apis.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'public_apis_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('public_apis.settings');

    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General Settings'),
      '#open' => TRUE,
    ];

    $form['general']['request_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request Timeout'),
      '#description' => $this->t('Maximum time (in seconds) to wait for API responses.'),
      '#default_value' => $config->get('request_timeout') ?: 30,
      '#min' => 5,
      '#max' => 300,
      '#step' => 1,
    ];

    $form['general']['cache_ttl'] = [
      '#type' => 'number',
      '#title' => $this->t('Cache TTL'),
      '#description' => $this->t('Time to live for cached API responses (in seconds).'),
      '#default_value' => $config->get('cache_ttl') ?: 3600,
      '#min' => 60,
      '#max' => 86400,
      '#step' => 60,
    ];

    $form['general']['enable_rate_limiting'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Rate Limiting'),
      '#description' => $this->t('Enable rate limiting to prevent API abuse.'),
      '#default_value' => $config->get('enable_rate_limiting') ?: TRUE,
    ];

    // Rate limiting settings.
    $form['rate_limiting'] = [
      '#type' => 'details',
      '#title' => $this->t('Rate Limiting Settings'),
      '#open' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="enable_rate_limiting"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['rate_limiting']['requests_per_minute'] = [
      '#type' => 'number',
      '#title' => $this->t('Requests per Minute'),
      '#default_value' => $config->get('requests_per_minute') ?: 60,
      '#min' => 1,
      '#max' => 1000,
    ];

    $form['rate_limiting']['requests_per_hour'] = [
      '#type' => 'number',
      '#title' => $this->t('Requests per Hour'),
      '#default_value' => $config->get('requests_per_hour') ?: 1000,
      '#min' => 1,
      '#max' => 10000,
    ];

    $form['rate_limiting']['requests_per_day'] = [
      '#type' => 'number',
      '#title' => $this->t('Requests per Day'),
      '#default_value' => $config->get('requests_per_day') ?: 10000,
      '#min' => 1,
      '#max' => 100000,
    ];

    // API Keys section.
    $form['api_keys'] = [
      '#type' => 'details',
      '#title' => $this->t('API Keys'),
      '#description' => $this->t('Configure API keys for services that require authentication.'),
      '#open' => TRUE,
    ];

    // Get all APIs that require authentication.
    $registry = $this->apiRegistry->getApiRegistry();
    $api_keys = $config->get('api_keys') ?: [];

    foreach ($registry as $category => $apis) {
      foreach ($apis as $api_slug => $api) {
        if ($api['auth_type'] !== 'none') {
          $form['api_keys'][$api_slug] = [
            '#type' => 'textfield',
            '#title' => $this->t('@name API Key', ['@name' => $api['name']]),
            '#description' => $this->t('API key for @name (@description)', [
              '@name' => $api['name'],
              '@description' => $api['description'],
            ]),
            '#default_value' => $api_keys[$api_slug] ?? '',
            '#maxlength' => 255,
          ];

          if ($api['auth_type'] === 'basic') {
            $form['api_keys'][$api_slug . '_username'] = [
              '#type' => 'textfield',
              '#title' => $this->t('@name Username', ['@name' => $api['name']]),
              '#default_value' => $config->get("api_credentials.{$api_slug}.username") ?: '',
            ];

            $form['api_keys'][$api_slug . '_password'] = [
              '#type' => 'password',
              '#title' => $this->t('@name Password', ['@name' => $api['name']]),
              '#description' => $this->t('Leave blank to keep current password.'),
            ];
          }
        }
      }
    }

    // Security settings.
    $form['security'] = [
      '#type' => 'details',
      '#title' => $this->t('Security Settings'),
      '#open' => FALSE,
    ];

    $form['security']['allowed_ips'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed IP Addresses'),
      '#description' => $this->t('List of IP addresses allowed to use the API endpoints (one per line). Leave empty to allow all IPs.'),
      '#default_value' => $config->get('allowed_ips') ?: '',
      '#rows' => 5,
    ];

    $form['security']['require_https'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Require HTTPS'),
      '#description' => $this->t('Require HTTPS for all API endpoints.'),
      '#default_value' => $config->get('require_https') ?: FALSE,
    ];

    // Logging settings.
    $form['logging'] = [
      '#type' => 'details',
      '#title' => $this->t('Logging Settings'),
      '#open' => FALSE,
    ];

    $form['logging']['log_api_calls'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log API Calls'),
      '#description' => $this->t('Log all API calls for debugging and monitoring.'),
      '#default_value' => $config->get('log_api_calls') ?: FALSE,
    ];

    $form['logging']['log_failed_calls'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log Failed Calls'),
      '#description' => $this->t('Log failed API calls for troubleshooting.'),
      '#default_value' => $config->get('log_failed_calls') ?: TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate IP addresses if provided.
    $allowed_ips = $form_state->getValue('allowed_ips');
    if (!empty($allowed_ips)) {
      $ips = array_filter(array_map('trim', explode("\n", $allowed_ips)));
      foreach ($ips as $ip) {
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
          $form_state->setErrorByName('allowed_ips', $this->t('Invalid IP address: @ip', ['@ip' => $ip]));
        }
      }
    }

    // Validate rate limiting values.
    $minute = $form_state->getValue('requests_per_minute');
    $hour = $form_state->getValue('requests_per_hour');
    $day = $form_state->getValue('requests_per_day');

    if ($minute * 60 > $hour) {
      $form_state->setErrorByName('requests_per_hour', $this->t('Requests per hour should be at least requests per minute × 60.'));
    }

    if ($hour * 24 > $day) {
      $form_state->setErrorByName('requests_per_day', $this->t('Requests per day should be at least requests per hour × 24.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('public_apis.settings');

    // Save general settings.
    $config->set('request_timeout', $form_state->getValue('request_timeout'))
      ->set('cache_ttl', $form_state->getValue('cache_ttl'))
      ->set('enable_rate_limiting', $form_state->getValue('enable_rate_limiting'))
      ->set('requests_per_minute', $form_state->getValue('requests_per_minute'))
      ->set('requests_per_hour', $form_state->getValue('requests_per_hour'))
      ->set('requests_per_day', $form_state->getValue('requests_per_day'))
      ->set('allowed_ips', $form_state->getValue('allowed_ips'))
      ->set('require_https', $form_state->getValue('require_https'))
      ->set('log_api_calls', $form_state->getValue('log_api_calls'))
      ->set('log_failed_calls', $form_state->getValue('log_failed_calls'));

    // Save API keys.
    $registry = $this->apiRegistry->getApiRegistry();
    $api_keys = [];
    $api_credentials = [];

    foreach ($registry as $category => $apis) {
      foreach ($apis as $api_slug => $api) {
        if ($api['auth_type'] !== 'none') {
          $key = $form_state->getValue($api_slug);
          if (!empty($key)) {
            $api_keys[$api_slug] = $key;
          }

          if ($api['auth_type'] === 'basic') {
            $username = $form_state->getValue($api_slug . '_username');
            $password = $form_state->getValue($api_slug . '_password');

            if (!empty($username)) {
              $api_credentials[$api_slug]['username'] = $username;
            }

            if (!empty($password)) {
              $api_credentials[$api_slug]['password'] = $password;
            }
          }
        }
      }
    }

    $config->set('api_keys', $api_keys);
    if (!empty($api_credentials)) {
      $config->set('api_credentials', $api_credentials);
    }

    $config->save();

    parent::submitForm($form, $form_state);
  }

}