<?php

namespace Drupal\public_apis\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for managing the API registry.
 */
class ApiRegistry {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs an ApiRegistry object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Gets the complete API registry.
   *
   * @return array
   *   The API registry organized by categories.
   */
  public function getApiRegistry(): array {
    // This is based on the public-apis repository structure.
    // In a real implementation, this could be stored in configuration
    // or loaded from a JSON file.
    return [
      'animals' => [
        'adoptapet' => [
          'name' => 'AdoptAPet',
          'base_url' => 'https://www.adoptapet.com/public/apis',
          'endpoints' => ['pet_list'],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Resource to help get pets adopted',
        ],
        'axolotl' => [
          'name' => 'Axolotl',
          'base_url' => 'https://theaxolotlapi.netlify.app',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Collection of axolotl pictures and facts',
        ],
        'cat_facts' => [
          'name' => 'Cat Facts',
          'base_url' => 'https://alexwohlbruck.github.io/cat-facts',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Daily cat facts',
        ],
        'cataas' => [
          'name' => 'Cataas',
          'base_url' => 'https://cataas.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Cat as a service (cats pictures and gifs)',
        ],
        'cats' => [
          'name' => 'Cats',
          'base_url' => 'https://docs.thecatapi.com',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Pictures of cats from Tumblr',
        ],
        'dog_facts' => [
          'name' => 'Dog Facts',
          'base_url' => 'https://dukengn.github.io/Dog-facts-API',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Random dog facts',
        ],
        'dogs' => [
          'name' => 'Dogs',
          'base_url' => 'https://dog.ceo/dog-api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Based on the Stanford Dogs Dataset',
        ],
        'ebird' => [
          'name' => 'eBird',
          'base_url' => 'https://documenter.getpostman.com/view/664302/S1ENwy59',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Retrieve recent or notable birding observations within a region',
        ],
        'fishwatch' => [
          'name' => 'FishWatch',
          'base_url' => 'https://www.fishwatch.gov/developers',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Information and pictures about individual fish species',
        ],
        'http_cat' => [
          'name' => 'HTTP Cat',
          'base_url' => 'https://http.cat',
          'endpoints' => ['{status_code}'],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Cat for every HTTP Status',
        ],
        'http_dog' => [
          'name' => 'HTTP Dog',
          'base_url' => 'https://http.dog',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Dogs for every HTTP response status code',
        ],
        'iucn' => [
          'name' => 'IUCN',
          'base_url' => 'http://apiv3.iucnredlist.org/api/v3/docs',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => FALSE,
          'cors' => FALSE,
          'description' => 'IUCN Red List of Threatened Species',
        ],
        'meowfacts' => [
          'name' => 'MeowFacts',
          'base_url' => 'https://github.com/wh-iterabb-it/meowfacts',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Get random cat facts',
        ],
        'movebank' => [
          'name' => 'Movebank',
          'base_url' => 'https://github.com/movebank/movebank-api-doc',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Movement and Migration data of animals',
        ],
        'petfinder' => [
          'name' => 'Petfinder',
          'base_url' => 'https://www.petfinder.com/developers',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Petfinder is dedicated to helping pets find homes, another resource to get pets adopted',
        ],
        'placebear' => [
          'name' => 'PlaceBear',
          'base_url' => 'https://placebear.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Placeholder bear pictures',
        ],
        'placedog' => [
          'name' => 'PlaceDog',
          'base_url' => 'https://place.dog',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Placeholder Dog pictures',
        ],
        'placekitten' => [
          'name' => 'PlaceKitten',
          'base_url' => 'https://placekitten.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Placeholder Kitten pictures',
        ],
        'randomdog' => [
          'name' => 'RandomDog',
          'base_url' => 'https://random.dog',
          'endpoints' => ['woof.json'],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Random pictures of dogs',
        ],
        'randomduck' => [
          'name' => 'RandomDuck',
          'base_url' => 'https://random-d.uk/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Random pictures of ducks',
        ],
        'randomfox' => [
          'name' => 'RandomFox',
          'base_url' => 'https://randomfox.ca/floof',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Random pictures of foxes',
        ],
        'rescuegroups' => [
          'name' => 'RescueGroups',
          'base_url' => 'https://userguide.rescuegroups.org/display/APIDG/API+Developers+Guide+Home',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Adoption',
        ],
        'shibe_online' => [
          'name' => 'Shibe.Online',
          'base_url' => 'http://shibe.online',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Random pictures of Shiba Inu, cats or birds',
        ],
        'the_dog' => [
          'name' => 'The Dog',
          'base_url' => 'https://thedogapi.com',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'A public service all about Dogs, free to use when making your fancy new App, Website or Service',
        ],
        'xeno_canto' => [
          'name' => 'xeno-canto',
          'base_url' => 'https://xeno-canto.org/explore/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Bird recordings',
        ],
        'zoo_animals' => [
          'name' => 'Zoo Animals',
          'base_url' => 'https://zoo-animal-api.herokuapp.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Facts and pictures of zoo animals',
        ],
      ],
      'anime' => [
        'aniapi' => [
          'name' => 'AniAPI',
          'base_url' => 'https://aniapi.com/docs',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Anime discovery, streaming & syncing with trackers',
        ],
        'anidb' => [
          'name' => 'AniDB',
          'base_url' => 'https://wiki.anidb.net/HTTP_API_Definition',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => FALSE,
          'cors' => FALSE,
          'description' => 'Anime Database',
        ],
        'anilist' => [
          'name' => 'AniList',
          'base_url' => 'https://github.com/AniList/ApiV2-GraphQL-Docs',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Anime discovery & tracking',
        ],
        'animechan' => [
          'name' => 'AnimeChan',
          'base_url' => 'https://github.com/RocktimSaikia/anime-chan',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Anime quotes (over 10k+)',
        ],
        'animefacts' => [
          'name' => 'AnimeFacts',
          'base_url' => 'https://chandan-02.github.io/anime-facts-rest-api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Anime Facts (over 100+)',
        ],
        'animenewsnetwork' => [
          'name' => 'AnimeNewsNetwork',
          'base_url' => 'https://www.animenewsnetwork.com/encyclopedia/api.php',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Anime industry news',
        ],
        'catboy' => [
          'name' => 'Catboy',
          'base_url' => 'https://catboys.com/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Neko images, funny GIFs & more',
        ],
        'danbooru_anime' => [
          'name' => 'Danbooru Anime',
          'base_url' => 'https://danbooru.donmai.us/wiki_pages/help:api',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Thousands of anime artist database to find good anime art',
        ],
        'jikan' => [
          'name' => 'Jikan',
          'base_url' => 'https://jikan.moe',
          'endpoints' => ['anime', 'manga', 'characters'],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Unofficial MyAnimeList API',
        ],
        'kitsu' => [
          'name' => 'Kitsu',
          'base_url' => 'https://kitsu.docs.apiary.io',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Anime discovery platform',
        ],
        'mangadex' => [
          'name' => 'MangaDex',
          'base_url' => 'https://api.mangadx.org/docs.html',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Manga Database and Community',
        ],
        'myanimelist' => [
          'name' => 'MyAnimeList',
          'base_url' => 'https://myanimelist.net/clubs.php?cid=13727',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Anime and Manga Database and Community',
        ],
        'nekosbest' => [
          'name' => 'NekosBest',
          'base_url' => 'https://docs.nekos.best',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Neko Images & Anime roleplaying GIFs',
        ],
        'shikimori' => [
          'name' => 'Shikimori',
          'base_url' => 'https://shikimori.one/api/doc',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Anime discovery, tracking, forum, rates',
        ],
        'studio_ghibli' => [
          'name' => 'Studio Ghibli',
          'base_url' => 'https://ghibliapi.herokuapp.com',
          'endpoints' => ['films', 'people', 'locations'],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Resources from Studio Ghibli films',
        ],
        'trace_moe' => [
          'name' => 'Trace Moe',
          'base_url' => 'https://soruly.github.io/trace.moe-api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'A useful tool to get the exact scene of an anime from a screenshot',
        ],
        'waifu_im' => [
          'name' => 'Waifu.im',
          'base_url' => 'https://waifu.im/docs',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Get waifu pictures from an archive of over 4000 images and multiple tags',
        ],
        'waifu_pics' => [
          'name' => 'Waifu.pics',
          'base_url' => 'https://waifu.pics/docs',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Image sharing platform for anime images',
        ],
      ],
      'anti_malware' => [
        'abuseipdb' => [
          'name' => 'AbuseIPDB',
          'base_url' => 'https://docs.abuseipdb.com',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'IP/domain/URL reputation',
        ],
        'alienvault_otx' => [
          'name' => 'AlienVault Open Threat Exchange (OTX)',
          'base_url' => 'https://otx.alienvault.com/api',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'IP/domain/URL reputation',
        ],
        'capesandbox' => [
          'name' => 'CAPEsandbox',
          'base_url' => 'https://capev2.readthedocs.io/en/latest/usage/api.html',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Malware execution and analysis',
        ],
        'google_safe_browsing' => [
          'name' => 'Google Safe Browsing',
          'base_url' => 'https://developers.google.com/safe-browsing',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Google Link/Domain Flagging',
        ],
        'maldatabase' => [
          'name' => 'MalDatabase',
          'base_url' => 'https://maldatabase.com/api-doc.html',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Provide malware datasets and threat intelligence feeds',
        ],
        'malshare' => [
          'name' => 'MalShare',
          'base_url' => 'https://malshare.com/doc.php',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Malware Archive / file sourcing',
        ],
        'malwarebazaar' => [
          'name' => 'MalwareBazaar',
          'base_url' => 'https://bazaar.abuse.ch/api',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Collect and share malware samples',
        ],
        'metacert' => [
          'name' => 'Metacert',
          'base_url' => 'https://metacert.com',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Metacert Link Flagging',
        ],
        'nophishy' => [
          'name' => 'NoPhishy',
          'base_url' => 'https://rapidapi.com/Amiichu/api/exerra-phishing-check',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Check links to see if they\'re known phishing attempts',
        ],
        'phisherman' => [
          'name' => 'Phisherman',
          'base_url' => 'https://phisherman.gg',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'IP/domain/URL reputation',
        ],
        'scanii' => [
          'name' => 'Scanii',
          'base_url' => 'https://docs.scanii.com',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Simple REST API that can scan submitted documents/files for the presence of threats',
        ],
        'urlhaus' => [
          'name' => 'URLhaus',
          'base_url' => 'https://urlhaus-api.abuse.ch',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Bulk queries and Download Malware Samples',
        ],
        'urlscan' => [
          'name' => 'URLScan.io',
          'base_url' => 'https://urlscan.io/about-api',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Scan and Analyse URLs',
        ],
        'virustotal' => [
          'name' => 'VirusTotal',
          'base_url' => 'https://www.virustotal.com/en/documentation/public-api',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'VirusTotal File/URL Analysis',
        ],
        'web_of_trust' => [
          'name' => 'Web of Trust',
          'base_url' => 'https://support.mywot.com/hc/en-us/sections/360004477734-API-',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'IP/domain/URL reputation',
        ],
      ],
      'art_design' => [
        'amethyste' => [
          'name' => 'Améthyste',
          'base_url' => 'https://api.amethyste.moe',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Generate images for Discord users',
        ],
        'art_institute_chicago' => [
          'name' => 'Art Institute of Chicago',
          'base_url' => 'https://api.artic.edu/docs',
          'endpoints' => ['artworks', 'exhibitions', 'artists'],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Art',
        ],
        'colormind' => [
          'name' => 'Colormind',
          'base_url' => 'http://colormind.io/api-access',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => FALSE,
          'cors' => FALSE,
          'description' => 'Color scheme generator',
        ],
        'colourlovers' => [
          'name' => 'ColourLovers',
          'base_url' => 'http://www.colourlovers.com/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => FALSE,
          'cors' => FALSE,
          'description' => 'Get various patterns, palettes and images',
        ],
        'cooper_hewitt' => [
          'name' => 'Cooper Hewitt',
          'base_url' => 'https://collection.cooperhewitt.org/api',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Smithsonian Design Museum',
        ],
        'dribbble' => [
          'name' => 'Dribbble',
          'base_url' => 'https://developer.dribbble.com',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Discover the world\'s top designers & creatives',
        ],
        'emojihub' => [
          'name' => 'EmojiHub',
          'base_url' => 'https://github.com/cheatsnake/emojihub',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Get emojis by categories and groups',
        ],
        'europeana' => [
          'name' => 'Europeana',
          'base_url' => 'https://pro.europeana.eu/resources/apis/search',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'European Museum and Galleries content',
        ],
        'harvard_art_museums' => [
          'name' => 'Harvard Art Museums',
          'base_url' => 'https://github.com/harvardartmuseums/api-docs',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => FALSE,
          'cors' => FALSE,
          'description' => 'Art',
        ],
        'icon_horse' => [
          'name' => 'Icon Horse',
          'base_url' => 'https://icon.horse',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Favicons for any website, with fallbacks',
        ],
        'iconfinder' => [
          'name' => 'Iconfinder',
          'base_url' => 'https://developer.iconfinder.com',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Icons',
        ],
        'icons8' => [
          'name' => 'Icons8',
          'base_url' => 'https://img.icons8.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Icons (find "search icon" hyperlink in page)',
        ],
        'lordicon' => [
          'name' => 'Lordicon',
          'base_url' => 'https://lordicon.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Icons with predone Animations',
        ],
        'met_museum' => [
          'name' => 'Metropolitan Museum of Art',
          'base_url' => 'https://metmuseum.github.io',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Met Museum of Art',
        ],
        'noun_project' => [
          'name' => 'Noun Project',
          'base_url' => 'http://api.thenounproject.com/index.html',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => FALSE,
          'cors' => FALSE,
          'description' => 'Icons',
        ],
        'php_noise' => [
          'name' => 'PHP-Noise',
          'base_url' => 'https://php-noise.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Noise Background Image Generator',
        ],
        'pixel_encounter' => [
          'name' => 'Pixel Encounter',
          'base_url' => 'https://pixelencounter.com/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'SVG Icon Generator',
        ],
        'rijksmuseum' => [
          'name' => 'Rijksmuseum',
          'base_url' => 'https://data.rijksmuseum.nl/object-metadata/api',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'RijksMuseum Data',
        ],
        'word_cloud' => [
          'name' => 'Word Cloud',
          'base_url' => 'https://wordcloudapi.com',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Easily create word clouds',
        ],
        'xcolors' => [
          'name' => 'xColors',
          'base_url' => 'https://x-colors.herokuapp.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Generate & convert colors',
        ],
      ],
      'books' => [
        'a_biblia_digital' => [
          'name' => 'A Bíblia Digital',
          'base_url' => 'https://www.abibliadigital.com.br/en',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Do not worry about managing the multiple versions of the Bible',
        ],
        'bhagavad_gita' => [
          'name' => 'Bhagavad Gita',
          'base_url' => 'https://docs.bhagavadgitaapi.in',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Open Source Shrimad Bhagavad Gita API including 21+ authors translation in Sanskrit/English/Hindi',
        ],
        'bhagavad_gita_io' => [
          'name' => 'Bhagavad Gita',
          'base_url' => 'https://bhagavadgita.io/api',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Bhagavad Gita text',
        ],
        'bhagavad_gita_telugu' => [
          'name' => 'Bhagavad Gita telugu',
          'base_url' => 'https://gita-api.vercel.app',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Bhagavad Gita API in telugu and odia languages',
        ],
        'bible_api' => [
          'name' => 'Bible-api',
          'base_url' => 'https://bible-api.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Free Bible API with multiple languages',
        ],
        'british_national_bibliography' => [
          'name' => 'British National Bibliography',
          'base_url' => 'http://bnb.data.bl.uk',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => FALSE,
          'cors' => FALSE,
          'description' => 'Books',
        ],
        'crossref_metadata_search' => [
          'name' => 'Crossref Metadata Search',
          'base_url' => 'https://github.com/CrossRef/rest-api-doc',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Books & Articles Metadata',
        ],
        'ganjoor' => [
          'name' => 'Ganjoor',
          'base_url' => 'https://api.ganjoor.net',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Classic Persian poetry works including access to related manuscripts, recitations and music tracks',
        ],
        'google_books' => [
          'name' => 'Google Books',
          'base_url' => 'https://developers.google.com/books',
          'endpoints' => [''],
          'auth_type' => 'OAuth',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Books',
        ],
        'gurbaninow' => [
          'name' => 'GurbaniNow',
          'base_url' => 'https://github.com/GurbaniNow/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Fast and Accurate Gurbani RESTful API',
        ],
        'gutendex' => [
          'name' => 'Gutendex',
          'base_url' => 'https://gutendex.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Web-API for fetching data from Project Gutenberg Books Library',
        ],
        'open_library' => [
          'name' => 'Open Library',
          'base_url' => 'https://openlibrary.org/developers/api',
          'endpoints' => ['books', 'search'],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Books, book covers and related data',
        ],
        'penguin_publishing' => [
          'name' => 'Penguin Publishing',
          'base_url' => 'http://www.penguinrandomhouse.biz/webservices/rest',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Books, book covers and related data',
        ],
        'poetrydb' => [
          'name' => 'PoetryDB',
          'base_url' => 'https://github.com/thundercomb/poetrydb#readme',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Enables you to get instant data from our vast poetry collection',
        ],
        'quran' => [
          'name' => 'Quran',
          'base_url' => 'https://quran.api-docs.io',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'RESTful Quran API with multiple languages',
        ],
        'quran_cloud' => [
          'name' => 'Quran Cloud',
          'base_url' => 'https://alquran.cloud/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'A RESTful Quran API to retrieve an Ayah, Surah, Juz or the entire Holy Quran',
        ],
        'quran_api' => [
          'name' => 'Quran-api',
          'base_url' => 'https://github.com/fawazahmed0/quran-api#readme',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Free Quran API Service with 90+ different languages and 400+ translations',
        ],
        'rig_veda' => [
          'name' => 'Rig Veda',
          'base_url' => 'https://aninditabasu.github.io/indica/html/rv.html',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Gods and poets, their categories, and the verse meters, with the mandal and sukta number',
        ],
        'the_bible' => [
          'name' => 'The Bible',
          'base_url' => 'https://docs.api.bible',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Everything you need from the Bible in one discoverable place',
        ],
        'thirukkural' => [
          'name' => 'Thirukkural',
          'base_url' => 'https://api-thirukkural.web.app',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => '1330 Thirukkural poems and explanation in Tamil and English',
        ],
        'vedic_society' => [
          'name' => 'Vedic Society',
          'base_url' => 'https://aninditabasu.github.io/indica/html/vs.html',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Descriptions of all nouns (names, places, animals, things) from vedic literature',
        ],
        'wizard_world' => [
          'name' => 'Wizard World',
          'base_url' => 'https://wizard-world-api.herokuapp.com/swagger/index.html',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Get information from the Harry Potter universe',
        ],
        'wolne_lektury' => [
          'name' => 'Wolne Lektury',
          'base_url' => 'https://wolnelektury.pl/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'API for obtaining information about e-books available on the WolneLektury.pl website',
        ],
      ],
      'weather' => [
        'weatherstack' => [
          'name' => 'Weatherstack',
          'base_url' => 'http://api.weatherstack.com/v1',
          'endpoints' => ['current', 'historical', 'forecast'],
          'auth_type' => 'apiKey',
          'auth_config' => [
            'location' => 'query',
            'name' => 'access_key',
          ],
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Weather information for any location',
        ],
        'openweathermap' => [
          'name' => 'OpenWeatherMap',
          'base_url' => 'https://openweathermap.org/api',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'auth_config' => [
            'location' => 'query',
            'name' => 'appid',
          ],
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Weather',
        ],
        'accuweather' => [
          'name' => 'AccuWeather',
          'base_url' => 'https://developer.accuweather.com/apis',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => FALSE,
          'cors' => FALSE,
          'description' => 'Weather and forecast data',
        ],
        'metaweather' => [
          'name' => 'MetaWeather',
          'base_url' => 'https://www.metaweather.com/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Weather',
        ],
      ],
      'finance' => [
        'fixer' => [
          'name' => 'Fixer',
          'base_url' => 'http://data.fixer.io/api',
          'endpoints' => ['latest', 'historical', 'symbols'],
          'auth_type' => 'apiKey',
          'auth_config' => [
            'location' => 'query',
            'name' => 'access_key',
          ],
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Foreign exchange rates',
        ],
        'alpha_vantage' => [
          'name' => 'Alpha Vantage',
          'base_url' => 'https://www.alphavantage.co',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Realtime and historical stock data',
        ],
        'coinapi' => [
          'name' => 'CoinAPI',
          'base_url' => 'https://docs.coinapi.io',
          'endpoints' => [''],
          'auth_type' => 'apiKey',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'All Currency Exchanges integrate under a single api',
        ],
        'coingecko' => [
          'name' => 'CoinGecko',
          'base_url' => 'http://www.coingecko.com/api',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Cryptocurrency Price, Market, and Developer/Social Data',
        ],
      ],
      'test_data' => [
        'jsonplaceholder' => [
          'name' => 'JSONPlaceholder',
          'base_url' => 'https://jsonplaceholder.typicode.com',
          'endpoints' => ['posts', 'comments', 'albums', 'photos', 'todos', 'users'],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => TRUE,
          'description' => 'Fake Online REST API for Testing and Prototyping',
        ],
        'fakestoreapi' => [
          'name' => 'FakeStoreAPI',
          'base_url' => 'https://fakestoreapi.com',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Fake store rest API for your e-commerce or shopping website prototype',
        ],
        'randomuser' => [
          'name' => 'RandomUser',
          'base_url' => 'https://randomuser.me',
          'endpoints' => [''],
          'auth_type' => 'none',
          'https' => TRUE,
          'cors' => FALSE,
          'description' => 'Generates and list user data',
        ],
      ],
    ];
  }

  /**
   * Gets all available categories.
   *
   * @return array
   *   Array of category names with metadata.
   */
  public function getCategories(): array {
    $registry = $this->getApiRegistry();
    $categories = [];

    foreach ($registry as $category => $apis) {
      $categories[$category] = [
        'name' => ucwords(str_replace('_', ' ', $category)),
        'slug' => $category,
        'apis' => array_keys($apis),
        'count' => count($apis),
      ];
    }

    return $categories;
  }

  /**
   * Gets APIs for a specific category.
   *
   * @param string $category
   *   The category slug.
   *
   * @return array
   *   Array of APIs in the category.
   */
  public function getApisByCategory(string $category): array {
    $registry = $this->getApiRegistry();
    return $registry[$category] ?? [];
  }

  /**
   * Gets configuration for a specific API.
   *
   * @param string $category
   *   The category slug.
   * @param string $api_name
   *   The API name.
   *
   * @return array|null
   *   The API configuration or NULL if not found.
   */
  public function getApiConfig(string $category, string $api_name): ?array {
    $registry = $this->getApiRegistry();
    return $registry[$category][$api_name] ?? NULL;
  }

  /**
   * Searches for APIs by name or description.
   *
   * @param string $query
   *   The search query.
   *
   * @return array
   *   Array of matching APIs.
   */
  public function searchApis(string $query): array {
    $registry = $this->getApiRegistry();
    $results = [];
    $query = strtolower($query);

    foreach ($registry as $category => $apis) {
      foreach ($apis as $api_slug => $api) {
        $name = strtolower($api['name']);
        $description = strtolower($api['description']);

        if (strpos($name, $query) !== FALSE || strpos($description, $query) !== FALSE) {
          $results[] = [
            'category' => $category,
            'api_slug' => $api_slug,
            'api' => $api,
          ];
        }
      }
    }

    return $results;
  }

}