<?php

namespace Drupal\public_apis\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\HttpFoundation\Response;

/**
 * Service for making API calls to public APIs.
 */
class PublicApisClient {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The rate limiter service.
   *
   * @var \Drupal\public_apis\Service\RateLimiter
   */
  protected $rateLimiter;

  /**
   * Constructs a PublicApisClient object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   */
  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    CacheBackendInterface $cache
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->cache = $cache;
  }

  /**
   * Set the rate limiter service.
   *
   * @param \Drupal\public_apis\Service\RateLimiter $rate_limiter
   *   The rate limiter service.
   */
  public function setRateLimiter(RateLimiter $rate_limiter): void {
    $this->rateLimiter = $rate_limiter;
  }

  /**
   * Makes an API call to a specified endpoint.
   *
   * @param string $category
   *   The API category.
   * @param string $api_name
   *   The API name.
   * @param string $endpoint
   *   The specific endpoint to call.
   * @param array $parameters
   *   Optional parameters for the API call.
   * @param string $method
   *   The HTTP method (GET, POST, etc.).
   *
   * @return array
   *   The API response data or error information.
   */
  public function makeApiCall(string $category, string $api_name, string $endpoint = '', array $parameters = [], string $method = 'GET'): array {
    $logger = $this->loggerFactory->get('public_apis');

    // Check rate limiting.
    if ($this->rateLimiter && !$this->rateLimiter->allowRequest($api_name)) {
      $logger->warning('Rate limit exceeded for API: @api', ['@api' => $api_name]);
      return [
        'success' => FALSE,
        'error' => 'Rate limit exceeded',
        'status_code' => 429,
      ];
    }

    // Check cache first for GET requests.
    $cache_key = "public_apis:{$category}:{$api_name}:" . md5($endpoint . serialize($parameters));
    if ($method === 'GET') {
      $cached = $this->cache->get($cache_key);
      if ($cached && !empty($cached->data)) {
        return $cached->data;
      }
    }

    try {
      // Get API configuration.
      $config = $this->configFactory->get('public_apis.settings');
      $api_config = $this->getApiConfig($category, $api_name);

      if (!$api_config) {
        throw new \Exception("API configuration not found for {$category}/{$api_name}");
      }

      // Build the full URL.
      $base_url = $api_config['base_url'];
      $url = rtrim($base_url, '/') . '/' . ltrim($endpoint, '/');

      // Prepare request options.
      $options = [
        'timeout' => $config->get('request_timeout') ?: 30,
        'headers' => [
          'Accept' => 'application/json',
          'User-Agent' => 'Drupal-PublicAPIs-Integration/1.0',
        ],
      ];

      // Add authentication if required.
      $auth_type = $api_config['auth_type'] ?? 'none';
      if ($auth_type !== 'none') {
        $this->addAuthentication($options, $api_config, $config);
      }

      // Add parameters.
      if (!empty($parameters)) {
        if ($method === 'GET') {
          $options['query'] = $parameters;
        }
        else {
          $options['json'] = $parameters;
        }
      }

      // Make the request.
      $response = $this->httpClient->request($method, $url, $options);
      $status_code = $response->getStatusCode();
      $body = $response->getBody()->getContents();

      // Decode JSON response.
      $data = json_decode($body, TRUE);
      if ($data === NULL && json_last_error() !== JSON_ERROR_NONE) {
        // If not JSON, return raw body.
        $data = ['raw_response' => $body];
      }

      $result = [
        'success' => TRUE,
        'data' => $data,
        'status_code' => $status_code,
        'headers' => $response->getHeaders(),
      ];

      // Cache successful GET responses.
      if ($method === 'GET' && $status_code === 200) {
        $cache_expire = time() + ($config->get('cache_ttl') ?: 3600);
        $this->cache->set($cache_key, $result, $cache_expire);
      }

      $logger->info('Successful API call to @api: @url', [
        '@api' => $api_name,
        '@url' => $url,
      ]);

      return $result;

    }
    catch (RequestException $e) {
      $status_code = $e->getResponse() ? $e->getResponse()->getStatusCode() : 0;
      $error_message = $e->getMessage();

      $logger->error('API call failed for @api: @error', [
        '@api' => $api_name,
        '@error' => $error_message,
      ]);

      return [
        'success' => FALSE,
        'error' => $error_message,
        'status_code' => $status_code,
      ];
    }
    catch (\Exception $e) {
      $logger->error('Unexpected error during API call to @api: @error', [
        '@api' => $api_name,
        '@error' => $e->getMessage(),
      ]);

      return [
        'success' => FALSE,
        'error' => $e->getMessage(),
        'status_code' => 500,
      ];
    }
  }

  /**
   * Adds authentication to the request options.
   *
   * @param array &$options
   *   The request options array.
   * @param array $api_config
   *   The API configuration.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The module configuration.
   */
  protected function addAuthentication(array &$options, array $api_config, $config): void {
    $auth_type = $api_config['auth_type'];
    $api_name = $api_config['name'];

    switch ($auth_type) {
      case 'apiKey':
        $api_key = $config->get("api_keys.{$api_name}");
        if ($api_key) {
          $key_location = $api_config['auth_config']['location'] ?? 'query';
          $key_name = $api_config['auth_config']['name'] ?? 'api_key';

          if ($key_location === 'header') {
            $options['headers'][$key_name] = $api_key;
          }
          else {
            $options['query'][$key_name] = $api_key;
          }
        }
        break;

      case 'bearer':
        $token = $config->get("api_keys.{$api_name}");
        if ($token) {
          $options['headers']['Authorization'] = "Bearer {$token}";
        }
        break;

      case 'basic':
        $username = $config->get("api_credentials.{$api_name}.username");
        $password = $config->get("api_credentials.{$api_name}.password");
        if ($username && $password) {
          $options['auth'] = [$username, $password];
        }
        break;
    }
  }

  /**
   * Gets the configuration for a specific API.
   *
   * @param string $category
   *   The API category.
   * @param string $api_name
   *   The API name.
   *
   * @return array|null
   *   The API configuration or NULL if not found.
   */
  protected function getApiConfig(string $category, string $api_name): ?array {
    // This would normally come from a configuration file or database.
    // For this example, we'll define some common APIs.
    $apis = $this->getApiRegistry();
    
    return $apis[$category][$api_name] ?? NULL;
  }

  /**
   * Gets the full API registry.
   *
   * @return array
   *   The API registry.
   */
  protected function getApiRegistry(): array {
    // This is a simplified registry. In a real implementation,
    // this would be stored in configuration or a separate service.
    return [
      'animals' => [
        'cat_facts' => [
          'name' => 'cat_facts',
          'base_url' => 'https://catfact.ninja',
          'auth_type' => 'none',
          'description' => 'Daily cat facts',
        ],
        'dog_facts' => [
          'name' => 'dog_facts',
          'base_url' => 'https://dog-facts-api.herokuapp.com/api/v1',
          'auth_type' => 'none',
          'description' => 'Random dog facts',
        ],
        'random_dog' => [
          'name' => 'random_dog',
          'base_url' => 'https://random.dog',
          'auth_type' => 'none',
          'description' => 'Random pictures of dogs',
        ],
      ],
      'weather' => [
        'weatherstack' => [
          'name' => 'weatherstack',
          'base_url' => 'http://api.weatherstack.com/v1',
          'auth_type' => 'apiKey',
          'auth_config' => [
            'location' => 'query',
            'name' => 'access_key',
          ],
          'description' => 'Weather information',
        ],
      ],
      'finance' => [
        'fixer' => [
          'name' => 'fixer',
          'base_url' => 'http://data.fixer.io/api',
          'auth_type' => 'apiKey',
          'auth_config' => [
            'location' => 'query',
            'name' => 'access_key',
          ],
          'description' => 'Foreign exchange rates',
        ],
      ],
    ];
  }

  /**
   * Gets all available API categories.
   *
   * @return array
   *   Array of categories with their APIs.
   */
  public function getCategories(): array {
    $registry = $this->getApiRegistry();
    $categories = [];

    foreach ($registry as $category => $apis) {
      $categories[$category] = [
        'name' => $category,
        'apis' => array_keys($apis),
        'count' => count($apis),
      ];
    }

    return $categories;
  }

  /**
   * Gets APIs for a specific category.
   *
   * @param string $category
   *   The category name.
   *
   * @return array
   *   Array of APIs in the category.
   */
  public function getApisByCategory(string $category): array {
    $registry = $this->getApiRegistry();
    return $registry[$category] ?? [];
  }

}