<?php

declare(strict_types=1);

namespace Drupal\publish_guard\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Publish Guard settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'publish_guard_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['publish_guard.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('publish_guard.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable publishing restrictions'),
      '#default_value' => $config->get('enabled'),
    ];

    $form['allowed_days'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed days'),
      '#options' => [
        0 => $this->t('Sunday'),
        1 => $this->t('Monday'),
        2 => $this->t('Tuesday'),
        3 => $this->t('Wednesday'),
        4 => $this->t('Thursday'),
        5 => $this->t('Friday'),
        6 => $this->t('Saturday'),
      ],
      '#default_value' => $config->get('allowed_days') ?? [],
      '#states' => [
        'visible' => [':input[name="enabled"]' => ['checked' => TRUE]],
      ],
    ];

    $form['allowed_start_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start time'),
      '#description' => $this->t('24-hour format (e.g., 09:00).'),
      '#default_value' => $config->get('allowed_start_time') ?? '09:00',
      '#size' => 8,
      '#maxlength' => 5,
      '#states' => [
        'visible' => [':input[name="enabled"]' => ['checked' => TRUE]],
      ],
    ];

    $form['allowed_end_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('End time'),
      '#description' => $this->t('24-hour format (e.g., 17:00).'),
      '#default_value' => $config->get('allowed_end_time') ?? '17:00',
      '#size' => 8,
      '#maxlength' => 5,
      '#states' => [
        'visible' => [':input[name="enabled"]' => ['checked' => TRUE]],
      ],
    ];

    $form['strictness'] = [
      '#type' => 'radios',
      '#title' => $this->t('Strictness'),
      '#options' => [
        'warn' => $this->t('Warn but allow publishing'),
        'block' => $this->t('Block publishing entirely'),
      ],
      '#default_value' => $config->get('strictness') ?? 'warn',
      '#states' => [
        'visible' => [':input[name="enabled"]' => ['checked' => TRUE]],
      ],
    ];

    $form['message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom message'),
      '#description' => $this->t('Leave blank for default.'),
      '#default_value' => $config->get('message') ?? '',
      '#states' => [
        'visible' => [':input[name="enabled"]' => ['checked' => TRUE]],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $start = $form_state->getValue('allowed_start_time');
    $end = $form_state->getValue('allowed_end_time');
    $pattern = '/^([01]\d|2[0-3]):[0-5]\d$/';

    if (!preg_match($pattern, $start)) {
      $form_state->setErrorByName('allowed_start_time', $this->t('Invalid time format.'));
    }
    if (!preg_match($pattern, $end)) {
      $form_state->setErrorByName('allowed_end_time', $this->t('Invalid time format.'));
    }
    if ($start >= $end) {
      $form_state->setErrorByName('allowed_end_time', $this->t('End time must be after start time.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $allowed_days = array_keys(array_filter($form_state->getValue('allowed_days')));

    $this->config('publish_guard.settings')
      ->set('enabled', (bool) $form_state->getValue('enabled'))
      ->set('allowed_days', array_map('intval', $allowed_days))
      ->set('allowed_start_time', $form_state->getValue('allowed_start_time'))
      ->set('allowed_end_time', $form_state->getValue('allowed_end_time'))
      ->set('strictness', $form_state->getValue('strictness'))
      ->set('message', $form_state->getValue('message'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
