<?php

declare(strict_types=1);

namespace Drupal\Tests\publish_guard\Kernel;

use Drupal\Core\Form\FormState;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * @group publish_guard
 */
class PublishGuardValidationTest extends KernelTestBase {

  protected static $modules = ['system', 'user', 'node', 'field', 'text', 'filter', 'publish_guard'];

  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installConfig(['system', 'filter', 'node', 'publish_guard']);
    $this->installSchema('node', ['node_access']);
    NodeType::create(['type' => 'article', 'name' => 'Article'])->save();
  }

  public function testBlockModeBlocksPublishing(): void {
    $this->configure(['enabled' => TRUE, 'allowed_days' => [], 'strictness' => 'block']);
    $this->setCurrentUser();

    $form_state = $this->createPublishFormState();
    publish_guard_node_form_validate([], $form_state);

    $this->assertTrue($form_state->hasAnyErrors());
    $this->assertArrayHasKey('status', $form_state->getErrors());
  }

  public function testWarnModeAllowsPublishing(): void {
    $this->configure(['enabled' => TRUE, 'allowed_days' => [], 'strictness' => 'warn']);
    $this->setCurrentUser();

    // Warn mode doesn't add the validator, so no errors.
    $form_state = $this->createPublishFormState();
    $this->assertFalse($form_state->hasAnyErrors());
  }

  public function testBypassPermissionAllowsPublishing(): void {
    $this->configure(['enabled' => TRUE, 'allowed_days' => [], 'strictness' => 'block']);
    $this->setCurrentUser(['bypass publish guard']);

    $form_state = $this->createPublishFormState();
    publish_guard_node_form_validate([], $form_state);

    $this->assertFalse($form_state->hasAnyErrors());
  }

  public function testDisabledModuleAllowsPublishing(): void {
    $this->configure(['enabled' => FALSE]);
    $this->setCurrentUser();

    $form_state = $this->createPublishFormState();
    publish_guard_node_form_validate([], $form_state);

    $this->assertFalse($form_state->hasAnyErrors());
  }

  public function testUnpublishedSaveAllowed(): void {
    $this->configure(['enabled' => TRUE, 'allowed_days' => [], 'strictness' => 'block']);
    $this->setCurrentUser();

    $form_state = new FormState();
    $form_state->setValue('status', ['value' => 0]);
    publish_guard_node_form_validate([], $form_state);

    $this->assertFalse($form_state->hasAnyErrors());
  }

  protected function configure(array $settings): void {
    $config = $this->config('publish_guard.settings');
    foreach ($settings as $key => $value) {
      $config->set($key, $value);
    }
    $config->save();
  }

  protected function setCurrentUser(array $permissions = []): void {
    $roles = [];
    if ($permissions) {
      $role = Role::load('test_role') ?? Role::create(['id' => 'test_role', 'label' => 'Test']);
      foreach ($permissions as $permission) {
        $role->grantPermission($permission);
      }
      $role->save();
      $roles = ['test_role'];
    }

    $user = User::create(['name' => $this->randomMachineName(), 'status' => 1, 'roles' => $roles]);
    $user->save();
    $this->container->get('current_user')->setAccount($user);
  }

  protected function createPublishFormState(): FormState {
    $form_state = new FormState();
    $form_state->setValue('status', ['value' => 1]);
    return $form_state;
  }

}
