<?php

declare(strict_types=1);

namespace Drupal\Tests\publish_guard\Unit;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\publish_guard\PublishGuardChecker;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\publish_guard\PublishGuardChecker
 * @group publish_guard
 */
class PublishGuardCheckerTest extends UnitTestCase {

  /**
   * @covers ::isPublishingAllowed
   * @dataProvider publishingAllowedProvider
   */
  public function testIsPublishingAllowed(array $config, string $datetime, bool $expected): void {
    $checker = $this->createChecker($config, strtotime($datetime . ' UTC'));
    $this->assertSame($expected, $checker->isPublishingAllowed());
  }

  /**
   * Data provider for testIsPublishingAllowed.
   */
  public static function publishingAllowedProvider(): array {
    $weekdays = ['enabled' => TRUE, 'allowed_days' => [1, 2, 3, 4, 5], 'allowed_start_time' => '09:00', 'allowed_end_time' => '17:00'];

    return [
      'disabled module' => [['enabled' => FALSE], '2025-01-06 12:00:00', TRUE],
      'within window' => [$weekdays, '2025-01-06 12:00:00', TRUE],
      'at start time' => [$weekdays, '2025-01-06 09:00:00', TRUE],
      'before start' => [$weekdays, '2025-01-06 08:00:00', FALSE],
      'at end time' => [$weekdays, '2025-01-06 17:00:00', FALSE],
      'after end' => [$weekdays, '2025-01-06 18:00:00', FALSE],
      'sunday blocked' => [$weekdays, '2025-01-05 12:00:00', FALSE],
    ];
  }

  /**
   * @covers ::getStrictnessLevel
   * @dataProvider strictnessProvider
   */
  public function testGetStrictnessLevel(?string $configured, string $expected): void {
    $checker = $this->createChecker(['strictness' => $configured]);
    $this->assertSame($expected, $checker->getStrictnessLevel());
  }

  /**
   * Data provider for testGetStrictnessLevel.
   */
  public static function strictnessProvider(): array {
    return [
      'warn' => ['warn', 'warn'],
      'block' => ['block', 'block'],
      'default' => [NULL, 'warn'],
    ];
  }

  /**
   * @covers ::getRestrictionMessage
   * @dataProvider messageProvider
   */
  public function testGetRestrictionMessage(?string $custom, string $expected): void {
    $checker = $this->createChecker(['message' => $custom]);
    $this->assertSame($expected, $checker->getRestrictionMessage());
  }

  /**
   * Data provider for testGetRestrictionMessage.
   */
  public static function messageProvider(): array {
    return [
      'custom' => ['No publishing!', 'No publishing!'],
      'empty' => ['', 'Publishing is currently restricted.'],
      'null' => [NULL, 'Publishing is currently restricted.'],
    ];
  }

  /**
   * Creates a checker with mocked dependencies.
   */
  protected function createChecker(array $config, ?int $time = NULL): PublishGuardChecker {
    $moduleConfig = $this->createMock(ImmutableConfig::class);
    $moduleConfig->method('get')->willReturnCallback(fn($key) => $config[$key] ?? NULL);

    $dateConfig = $this->createMock(ImmutableConfig::class);
    $dateConfig->method('get')->with('timezone.default')->willReturn('UTC');

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory->method('get')->willReturnCallback(
      fn($name) => $name === 'publish_guard.settings' ? $moduleConfig : $dateConfig
    );

    $timeService = $this->createMock(TimeInterface::class);
    $timeService->method('getCurrentTime')->willReturn($time ?? time());

    return new PublishGuardChecker($configFactory, $timeService);
  }

}
