<?php

declare(strict_types=1);

namespace Drupal\publish_guard;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Checks if publishing is allowed based on configured time windows.
 */
class PublishGuardChecker {

  use StringTranslationTrait;

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected TimeInterface $time,
  ) {}

  /**
   * Checks if publishing is currently allowed.
   *
   * @return bool
   *   TRUE if publishing is allowed, FALSE if restricted.
   */
  public function isPublishingAllowed(): bool {
    $config = $this->configFactory->get('publish_guard.settings');

    if (!$config->get('enabled')) {
      return TRUE;
    }

    $timezone = new \DateTimeZone(
      $this->configFactory->get('system.date')->get('timezone.default') ?? 'UTC'
    );
    $now = (new \DateTime('@' . $this->time->getCurrentTime()))->setTimezone($timezone);

    // Check day (0 = Sunday, 6 = Saturday).
    if (!in_array((int) $now->format('w'), $config->get('allowed_days') ?? [], TRUE)) {
      return FALSE;
    }

    // Check time window (string comparison of H:i format is safe).
    $current_time = $now->format('H:i');
    $start = $config->get('allowed_start_time') ?? '09:00';
    $end = $config->get('allowed_end_time') ?? '17:00';

    return $current_time >= $start && $current_time < $end;
  }

  /**
   * Gets the strictness level.
   *
   * @return string
   *   The strictness level: 'warn' or 'block'.
   */
  public function getStrictnessLevel(): string {
    return $this->configFactory->get('publish_guard.settings')->get('strictness') ?? 'warn';
  }

  /**
   * Gets the restriction message to display.
   *
   * @return string
   *   The message to display when publishing is restricted.
   */
  public function getRestrictionMessage(): string {
    $message = $this->configFactory->get('publish_guard.settings')->get('message');
    return $message ?: (string) $this->t('Publishing is currently restricted.');
  }

}
