<?php

namespace Drupal\purge_users\Entity;

use Drupal\Core\Condition\ConditionPluginCollection;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\purge_users\PurgeUsersPolicyInterface;

/**
 * Defines a Purge Users Policy configuration entity class.
 *
 * @ConfigEntityType(
 *   id = "purge_users_policy",
 *   label = @Translation("Purge Users Policy"),
 *   label_singular = @Translation("Purge Users Policy"),
 *   label_plural = @Translation("Purge Users Policies"),
 *   label_count = @PluralTranslation(
 *     singular = @Translation("purge user policy"),
 *     plural = @Translation("purge user policies"),
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\purge_users\Controller\PurgeUsersPolicyListBuilder",
 *     "form" = {
 *       "add" = "Drupal\purge_users\Form\PurgeUsersPolicyAddForm",
 *       "delete" = "Drupal\purge_users\Form\PurgeUsersPolicyDeleteForm",
 *       "edit" = "Drupal\purge_users\Form\PurgeUsersPolicyEditForm"
 *     }
 *   },
 *   static_cache = TRUE,
 *   translatable = TRUE,
 *   config_prefix = "purge_users_policy",
 *   admin_permission = "administer site configuration",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label"
 *   },
 *   links = {
 *     "add-form" = "/admin/config/people/purge-users/policies/add",
 *     "edit-form" = "/admin/config/people/purge-users/policies/{purge_users_policy}",
 *     "delete-form" = "/admin/config/people/purge-users/policies/delete/{purge_users_policy}",
 *     "collection" = "/admin/config/people/purge-users/policies"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "policy_conditions",
 *     "purge_user_cancel_method",
 *     "disregard_blocked_users",
 *     "purge_on_cron",
 *     "inactive_user_notify_subject",
 *     "inactive_user_notify_text",
 *     "send_email_notification",
 *     "user_before_deletion_subject",
 *     "user_before_deletion_text",
 *     "send_email_user_before_notification",
 *     "user_before_notification_value",
 *     "user_before_notification_period"
 *   }
 * )
 */
class PurgeUsersPolicy extends ConfigEntityBase implements PurgeUsersPolicyInterface {

  /**
   * The ID of the purge user policy.
   *
   * @var int
   */
  protected $id;

  /**
   * The policy title.
   *
   * @var string
   */
  protected $label;

  /**
   * Condition instance IDs.
   *
   * @var array
   */
  protected $policy_conditions = [];

  /**
   * The conditions as a collection.
   *
   * @var \Drupal\Core\Plugin\DefaultLazyPluginCollection
   */
  protected $conditionsCollection;

  /**
   * The selected user cancel method.
   *
   * @var string|null
   */
  protected $purge_user_cancel_method;

  /**
   * Disregard inactive/blocked users.
   *
   * @var bool|null
   */
  protected $purge_on_cron;

  /**
   * Purge on cron.
   *
   * @var bool|null
   */
  protected $disregard_blocked_users;

  /**
   * The deletion email subject.
   *
   * @var string|null
   */
  protected $inactive_user_notify_subject;

  /**
   * The deletion email text.
   *
   * @var string|null
   */
  protected $inactive_user_notify_text;

  /**
   * Send email on deletion.
   *
   * @var string|null
   */
  protected $send_email_notification;

  /**
   * The before deletion mail subject.
   *
   * @var string|null
   */
  protected $user_before_deletion_subject;

  /**
   * The before deletion mail text.
   *
   * @var string|null
   */
  protected $user_before_deletion_text;

  /**
   * Send notification email before deletion.
   *
   * @var string|null
   */
  protected $send_email_user_before_notification;

  /**
   * The notification delay value.
   *
   * @var int|null
   */
  protected $user_before_notification_value;

  /**
   * The notification delay period type.
   *
   * @var string|null
   */
  protected $user_before_notification_period;

  /**
   * {@inheritdoc}
   */
  public function id() {
    return $this->id;
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->label;
  }

  /**
   * {@inheritdoc}
   */
  public function getConditions() {
    return $this->policy_conditions;
  }

  /**
   * {@inheritdoc}
   */
  public function getCondition($key) {
    if (!isset($this->policy_conditions[$key])) {
      return NULL;
    }
    return $this->policy_conditions[$key];
  }

  /**
   * Get the selected user cancel method.
   */
  public function getPurgeUserCancelMethod(): string|null {
    return $this->purge_user_cancel_method;
  }

  /**
   * Get disregard blocked users status.
   */
  public function isDisregardBlockedUsers(): bool|null {
    return $this->disregard_blocked_users;
  }

  /**
   * Get purge on cron status.
   */
  public function isPurgeOnCron(): bool|null {
    return $this->purge_on_cron;
  }

  /**
   * Get the deletion email subject.
   */
  public function getInactiveUserNotifySubject(): string|null {
    return $this->inactive_user_notify_subject;
  }

  /**
   * Get the deletion email text.
   */
  public function getInactiveUserNotifyText(): string|null {
    return $this->inactive_user_notify_text;
  }

  /**
   * Get the email notification on user deletion status.
   */
  public function getSendEmailNotification(): string|null {
    return $this->send_email_notification;
  }

  /**
   * Get the before deletion mail subject.
   */
  public function getUserBeforeDeletionSubject(): string|null {
    return $this->user_before_deletion_subject;
  }

  /**
   * Get the before deletion mail text.
   */
  public function getUserBeforeDeletionText(): string|null {
    return $this->user_before_deletion_text;
  }

  /**
   * Get the email notification before user deletion status.
   */
  public function getSendEmailUserBeforeNotification(): string|null {
    return $this->send_email_user_before_notification;
  }

  /**
   * Get the notification delay value.
   */
  public function getUserBeforeNotificationValue(): int|null {
    return $this->user_before_notification_value;
  }

  /**
   * Get the notification delay period type.
   */
  public function getUserBeforeNotificationPeriod(): string|null {
    return $this->user_before_notification_period;
  }

  /**
   * Get the plugin collections used by this entity.
   *
   * @return \Drupal\Core\Condition\ConditionPluginCollection
   *   An array of plugin collections, keyed by the property name they use to
   *   store their configuration.
   */
  public function getConditionsCollection() {
    if (!isset($this->conditionsCollection)) {
      $this->conditionsCollection = new ConditionPluginCollection(\Drupal::service('plugin.manager.condition'), $this->getConditions());
    }
    return $this->conditionsCollection;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    parent::calculateDependencies();
    $conditions_collection = $this->getConditionsCollection();
    if (empty($conditions_collection)) {
      return $this;
    }
    $condition_plugin_ids = $conditions_collection->getInstanceIds();
    foreach ($condition_plugin_ids as $condition_plugin_id) {
      $condition_plugin = $conditions_collection->get($condition_plugin_id);
      $condition_plugin_dependencies = $this->getPluginDependencies($condition_plugin);
      $this->addDependencies($condition_plugin_dependencies);
    }
    return $this;
  }

}
