<?php

namespace Drupal\purge_users\Form;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Editing a condition within the policy form.
 */
class ConditionEdit extends FormBase {

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Plugin manager of the purge users policy conditions.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $manager;

  /**
   * The Purge Users Policy entity.
   *
   * @var \Drupal\purge_users\Entity\PurgeUsersPolicy
   */
  protected $policy;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.condition'),
      $container->get('current_route_match'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Overriding the constructor to load in the plugin manager.
   *
   * @param \Drupal\Component\Plugin\PluginManagerInterface $manager
   *   The plugin manager for the purge users policy conditions.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   *   The plugin manager for the purge users policy conditions.
   */
  public function __construct(PluginManagerInterface $manager, RouteMatchInterface $route_match, EntityTypeManagerInterface $entity_type_manager) {
    $this->manager = $manager;
    $this->routeMatch = $route_match;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Returns a unique string identifying the form.
   *
   * @return string
   *   The unique string identifying the form.
   */
  public function getFormId() {
    return 'purge_users_policy_condition_edit_form';
  }

  /**
   * Form constructor.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param string $policy_id
   *   ID of the purge_users_policy entity.
   * @param string $plugin_id
   *   The condition plugin id.
   * @param string $id
   *   The ID of the condition.
   *
   * @return array
   *   The form structure.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $policy_id = NULL, $plugin_id = NULL, $id = NULL) {
    if (empty($plugin_id)) {
      // Should not happen, but just in case.
      return [];
    }
    // Set the policy entity.
    $this->policy = $this->routeMatch->getParameter('purge_users_policy');
    if (!$this->policy) {
      $policy_storage = $this->entityTypeManager->getStorage('purge_users_policy');
      $this->policy = $policy_storage->loadByProperties(['id' => $policy_id])[$policy_id];
    }
    $condition = $this->policy->getCondition($id);
    if ($condition) {
      $instance = $this->manager->createInstance($condition['id'], $condition);
    }
    else {
      $instance = $this->manager->createInstance($plugin_id, []);
    }
    /** @var \Drupal\purge_users\PurgeUsersConditionInterface $instance */
    $form = $instance->buildConfigurationForm($form, $form_state);
    $form['id'] = [
      '#type' => 'value',
      // The default condition id is the plugin id.
      '#value' => $id ?: $plugin_id,
    ];
    $form['instance'] = [
      '#type' => 'value',
      '#value' => $instance,
    ];
    $form['#title'] = $instance->getLabel();
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#ajax' => [
        'callback' => [$this, 'ajaxSave'],
      ],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $instance = $form_state->getValue('instance');
    $instance->validateConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $conditions = $this->policy->getConditions();
    /** @var \Drupal\purge_users\PurgeUsersConditionInterface $instance */
    $instance = $form_state->getValue('instance');
    $instance->submitConfigurationForm($form, $form_state);
    if ($form_state->hasValue('id')) {
      $conditions[$form_state->getValue('id')] = $instance->getConfiguration();
    }
    else {
      $conditions[] = $instance->getConfiguration();
    }
    $this->policy->set('policy_conditions', $conditions);
    $this->policy->save();
  }

  /**
   * {@inheritdoc}
   */
  public function ajaxSave(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $url = Url::fromRoute('entity.purge_users_policy.edit_form', ['purge_users_policy' => $this->policy->id()]);
    $response->addCommand(new RedirectCommand($url->toString()));
    $response->addCommand(new CloseModalDialogCommand());
    return $response;
  }

}
