<?php

namespace Drupal\purge_users\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\purge_users\Entity\PurgeUsersPolicy;
use Drupal\purge_users\Plugin\BatchWorker\BatchWorker;
use Drupal\purge_users\Services\PurgeUsersPolicyServiceInterface;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Form to purge and/or notify users through the UI with policies.
 *
 * @package Drupal\purge_users\Form
 */
class PolicyConfirmationForm extends ConfirmFormBase {
  /**
   * The number of users to show before resorting to "... and x more.".
   */
  const NUMBER_OF_USERS_TO_SHOW = 50;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The purge users policy service.
   *
   * @var \Drupal\purge_users\Services\PurgeUsersPolicyServiceInterface
   */
  protected $policyService;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The policy used for UI purge.
   *
   * @var \Drupal\purge_users\Entity\PurgeUsersPolicy
   */
  protected $policy;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    PurgeUsersPolicyServiceInterface $policy_service,
    MessengerInterface $messenger,
  ) {
    $this->configFactory = $config_factory;
    $this->policyService = $policy_service;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('purge_users.policy_service'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'purge_users_policy_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Purge user confirmation');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.purge_users_policy.edit_form', ['purge_users_policy' => $this->policy->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('Are you sure you want to cancel these user accounts?');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Confirm');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText() {
    return $this->t('Cancel');
  }

  /**
   * Sets the policy.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy|null $policy
   *   The policy to set.
   *
   * @see KernelTestBase::runPurgeOperation()
   */
  public function setPolicy($policy = NULL) {
    $this->policy = $policy;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $policy_id = NULL) {
    $this->setPolicy(PurgeUsersPolicy::load($policy_id));
    $form = parent::buildForm($form, $form_state);
    $uids = $this->policyService->getPolicyUserIdsToPurge($this->policy);
    $form['accounts'] = [
      '#prefix' => '<ul>',
      '#suffix' => '</ul>',
      '#tree' => TRUE,
    ];

    $uidsToShow = array_slice($uids, 0, self::NUMBER_OF_USERS_TO_SHOW);
    $accounts = User::loadMultiple($uidsToShow);
    foreach ($accounts as $account) {
      // Prevent user 1 from being canceled.
      if ($account->get('uid')->value <= 1) {
        continue;
      }
      $form['accounts']['uid' . $account->get('uid')->value] = [
        '#type' => 'markup',
        '#value' => $account->get('uid')->value,
        '#prefix' => '<li>',
        '#suffix' => $account->get('name')->value . " &lt;" . $account->get('mail')->value . "&gt; </li>\n",
      ];
    }

    if (count($uids) > self::NUMBER_OF_USERS_TO_SHOW) {
      $form['accounts']['and_more'] = [
        '#type' => 'markup',
        '#markup' => $this->t('...and @more more.', ['@more' => count($uids) - self::NUMBER_OF_USERS_TO_SHOW]),
        '#prefix' => '<li>',
        '#suffix' => '</li>',
      ];
    }

    $form['dry_run'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Dry run'),
      '#description' => $this->t('Simulate the purge without deleting or canceling users'),
      '#default_value' => FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $ids = $this->policyService->getPolicyUserIdsToPurge($this->policy);
    $notify_ids = $this->policyService->getPolicyUserIdsToNotify($this->policy);
    if (!$ids && !$notify_ids) {
      // Nothing to do.
      return;
    }
    // Initialize a batch operation.
    $batch = [
      'operations' => [],
      'finished' => [$this, 'batchCompleted'],
      'title' => $this->t('Delete users'),
      'init_message' => $this->t('Delete users operation is starting...'),
      'progress_message' => $this->t('Processed @current out of @total.'),
      'error_message' => $this->t('Delete users operation has encountered an error.'),
    ];
    $dry_run = (bool) $form_state->getValue('dry_run') ?? FALSE;
    // User load multiple to process through batch operation.
    foreach ($ids as $id) {
      // phpcs:ignore
      $batch['operations'][] = [[BatchWorker::class, 'batchWorkerPurgeUsers'], [$id, $this->policy, $dry_run]];
    }
    // Prenotification batch.
    foreach ($notify_ids as $id) {
      // phpcs:ignore
      $batch['operations'][] = [[BatchWorker::class, 'batchWorkerNotifyUsers'], [$id, $this->policy]];
    }
    // Batch set.
    batch_set($batch);
  }

  /**
   * The batch finish handler.
   */
  public function batchCompleted($success, $results, $operations) {
    if ($success) {
      $purged = $results['purged'] ?? 0;
      $message = t('@count users have been purged.', ['@count' => $purged]);
      $this->messenger->addMessage($message);
    }
    else {
      $error_operation = reset($operations);
      $message = t('An error occurred while processing %error_operation with arguments: @arguments', [
        '%error_operation' => $error_operation[0],
        '@arguments' => print_r($error_operation[1], TRUE),
      ]);
      $this->messenger->addError($message);
    }
    return new RedirectResponse(Url::fromRoute(
      'entity.purge_users_policy.edit_form',
      ['purge_users_policy' => $this->policy->id()]
    )->toString());
  }

}
