<?php

namespace Drupal\purge_users\Form;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\purge_users\Entity\PurgeUsersPolicy;
use Drupal\purge_users\Services\PurgeUsersPolicyServiceInterface;
use Drupal\user\RoleStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form that lists out the conditions for the policy.
 */
class PurgeUsersPolicyEditForm extends PurgeUsersPolicyForm {

  /**
   * Role storage.
   *
   * @var \Drupal\user\RoleStorageInterface
   */
  protected $storage;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The messenger.
   *
   * @var \Drupal\purge_users\Services\PurgeUsersPolicyServiceInterface
   */
  protected $policyService;

  /**
   * Overridden constructor to load the plugin.
   *
   * @param \Drupal\Component\Plugin\PluginManagerInterface $manager
   *   Plugin manager for conditions.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\user\RoleStorageInterface $storage
   *   Role storage.
   * @param \Drupal\purge_users\Services\PurgeUsersPolicyServiceInterface $policy_service
   *   The purge users policy service.
   */
  public function __construct(
    PluginManagerInterface $manager,
    LanguageManagerInterface $language_manager,
    FormBuilderInterface $form_builder,
    MessengerInterface $messenger,
    RoleStorageInterface $storage,
    PurgeUsersPolicyServiceInterface $policy_service,
  ) {
    parent::__construct($manager, $language_manager, $form_builder);
    $this->messenger = $messenger;
    $this->storage = $storage;
    $this->policyService = $policy_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $entity_type_manager = $container->get('entity_type.manager');
    return new static(
      $container->get('plugin.manager.condition'),
      $container->get('language_manager'),
      $container->get('form_builder'),
      $container->get('messenger'),
      $entity_type_manager->getStorage('user_role'),
      $container->get('purge_users.policy_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\purge_users\Entity\PurgeUsersPolicy $policy */
    $policy = $this->entity;
    $form = parent::form($form, $form_state);
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    $available_conditions = [];
    foreach ($this->manager->getDefinitions() as $plugin_id => $definition) {
      if ($policy->getCondition($plugin_id)) {
        continue;
      }
      if (isset($definition['context_definitions']['context'])) {
        $available_conditions[$plugin_id] = (string) $definition['label'];
      }
    }
    // Sort ascending by condition label.
    asort($available_conditions);

    $form['conditions_fieldset'] = [
      '#type' => 'fieldset',
    ];

    $form['conditions_fieldset']['condition_list'] = [
      '#markup' => '<h2>' . $this->t('Policy conditions') . '</h2>',
    ];

    $form['conditions_fieldset']['condition_container'] = [
      '#type' => 'container',
    ];
    $form['conditions_fieldset']['condition_container']['condition'] = [
      '#type' => 'select',
      '#options' => $available_conditions ?: [$this->t('No more conditions available')],
      '#disabled' => empty($available_conditions),
    ];
    $form['conditions_fieldset']['condition_container']['add'] = [
      '#type' => 'submit',
      '#name' => 'add',
      '#value' => $this->t('Add condition'),
      '#ajax' => [
        'callback' => [$this, 'add'],
      ],
      '#disabled' => empty($available_conditions),
    ];

    $condition_instances = $this->getPolicyConditionsInstances($policy);
    if (!$this->hasStatusRelatedCondition($condition_instances)) {
      $form['conditions_fieldset']['warning_message'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        'message' => [
          '#markup' => $this->t('At least one condition related to user inactivity is required to enable purging (e.g., never logged in, blocked, or inactive since a given date).'),
        ],
      ];
    }

    $form['conditions_fieldset']['items'] = [
      '#type' => 'markup',
      '#prefix' => '<div id="configured-conditions">',
      '#suffix' => '</div>',
      '#theme' => 'table',
      '#header' => [
        'condition' => $this->t('Condition'),
        'summary' => $this->t('Summary'),
        'operations' => $this->t('Operations'),
      ],
      '#rows' => $this->renderRows($policy, $condition_instances),
      '#empty' => $this->t('No conditions have been configured.'),
    ];

    $form['conditions_fieldset']['condition_list_footer'] = [
      '#markup' => '<p>' . $this->t('<strong>Note:</strong> Users must meet all conditions to be eligible for purging. <em>(Conditions are combined with AND logic.)</em></p>'),
    ];

    $notification_subject = $this->t('Your account is deleted');
    $notification_users_before_subject = $this->t('Your account will be deleted');
    $notification_text = $this->t("Dear User, \n\nYour account has been deleted due the website’s policy to automatically remove users who match certain criteria. If you have concerns regarding the deletion, please talk to the administrator of the website.\n\nThank you");
    $notification_users_before_text = $this->t("Dear User, \n\nYour account will be deleted soon due the website’s policy to automatically remove users who match certain criteria. If you have concerns regarding the deletion, please talk to the administrator of the website.\n\nThank you");

    $user_cancel_methods = user_cancel_methods();
    // Add the possibility to choose the site policy.
    $user_cancel_methods['#options']['user_cancel_site_policy'] = $this->t("Follow site's policy");
    // Add the possibility to choose the module's default.
    $user_cancel_methods['#options'] = array_merge(
      ['default' => $this->t("Use module's default method")], $user_cancel_methods['#options']
    );

    $form['purge_user_cancel_method'] = [
      '#type' => 'radios',
      '#required' => TRUE,
      '#title' => $this->t('When cancelling a user account'),
      '#default_value' => $this->entity->getPurgeUserCancelMethod() ?: 'default',
      '#options' => $user_cancel_methods['#options'],
    ];

    $form['disregard_blocked_users'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disregard inactive/blocked users'),
      '#default_value' => $this->entity->isDisregardBlockedUsers() ?? FALSE,
      '#description' => $this->t('Do not look at inactive and blocked users. If you use a cancellation method that blocks users, this should normally be enabled, or blocked users will be included in the processing and make it unnecessarily heavy.'),
    ];

    $form['purge_on_cron'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Purge on cron'),
      '#default_value' => $this->entity->isPurgeOnCron() ?? FALSE,
    ];

    $form['user_notification'] = [
      '#type' => 'details',
      '#title' => $this->t('User Deletion Notification'),
      '#open' => FALSE,
    ];

    $form['user_notification']['send_email_notification'] = [
      '#type' => 'radios',
      '#required' => TRUE,
      '#title' => $this->t('Send email notification to purged users'),
      '#default_value' => $this->entity->getSendEmailNotification() ?? 'default',
      '#options' => [
        'default' => $this->t("Use module's default settings"),
        'enabled' => $this->t('Enable'),
        'disabled' => $this->t('Disable'),
      ],
    ];

    $form['user_notification']['inactive_user_notify_subject'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Subject of user notification email'),
      '#default_value' => $this->entity->getInactiveUserNotifySubject() ?: $notification_subject,
      '#cols' => 1,
      '#rows' => 1,
      '#description' => $this->t('Customize the subject of the notification email sent to the user.'),
      '#required' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="send_email_notification"]' => ['value' => 'enabled'],
        ],
      ],
    ];

    $form['user_notification']['inactive_user_notify_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body of user notification email'),
      '#default_value' => $this->entity->getInactiveUserNotifyText() ?: $notification_text,
      '#cols' => 70,
      '#rows' => 10,
      '#description' => $this->t('Customize the body of the notification email sent to the user.'),
      '#required' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="send_email_notification"]' => ['value' => 'enabled'],
        ],
      ],
    ];

    $form['user_before_deletion'] = [
      '#type' => 'details',
      '#title' => $this->t('Notification User Before Deletion'),
      '#open' => FALSE,
    ];

    $form['user_before_deletion']['send_email_user_before_notification'] = [
      '#type' => 'radios',
      '#required' => TRUE,
      '#title' => $this->t('Send email notification before purging users'),
      '#default_value' => $this->entity->getSendEmailUserBeforeNotification() ?? 'default',
      '#options' => [
        'default' => $this->t("Use module's default settings"),
        'enabled' => $this->t('Enable'),
        'disabled' => $this->t('Disable'),
      ],
    ];

    $form['user_before_deletion']['user_before_deletion_subject'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Subject of user before deletion notification email'),
      '#default_value' => $this->entity->getUserBeforeDeletionSubject() ?: $notification_users_before_subject,
      '#cols' => 1,
      '#rows' => 1,
      '#description' => $this->t('Customize the subject of the notification email sent to the user.'),
      '#required' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="send_email_user_before_notification"]' => ['value' => 'enabled'],
        ],
      ],
    ];

    $form['user_before_deletion']['user_before_deletion_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body of user before deletion notification email'),
      '#default_value' => $this->entity->getUserBeforeDeletionText() ?: $notification_users_before_text,
      '#cols' => 70,
      '#rows' => 10,
      '#description' => $this->t('Customize the body of the notification email sent to the user.'),
      '#required' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="send_email_user_before_notification"]' => ['value' => 'enabled'],
        ],
      ],
    ];

    $form['user_before_deletion']['notification_period'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Notification period'),
      '#description' => $this->t('Specify how long before account cancellation the notification should be sent.'),
      '#attributes' => [
        'class' => ['purge-interval-selector', 'clearfix'],
      ],
      '#states' => [
        'visible' => [
          ':input[name="send_email_user_before_notification"]' => ['value' => 'enabled'],
        ],
      ],
    ];

    $form['user_before_deletion']['notification_period']['user_before_notification_value'] = [
      '#type' => 'number',
      '#title' => $this->t('Value'),
      '#default_value' => $this->entity->getUserBeforeNotificationValue() ?? 0,
      '#attributes' => ['class' => ['purge-value']],
    ];

    $form['user_before_deletion']['notification_period']['user_before_notification_period'] = [
      '#type' => 'select',
      '#title' => $this->t('Period'),
      '#options' => [
        'days' => $this->t('Days'),
        'month' => $this->t('Months'),
        'year' => $this->t('Year'),
      ],
      '#default_value' => $this->entity->getUserBeforeNotificationPeriod() ?? 'days',
      '#attributes' => ['class' => ['purge-period']],
    ];

    if ($this->moduleHandler->moduleExists('token')) {
      $form['token_help'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['user'],
        '#show_restricted' => TRUE,
        '#show_nested' => FALSE,
      ];
    }

    // Attach library.
    $form['#attached']['library'][] = 'purge_users/styling';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);

    $actions['purge_users_now'] = [
      '#type' => 'submit',
      '#value' => $this->t('Purge users now'),
      '#attributes' => [
        'class' => [
          'purge-now',
          'button button--primary',
        ],
      ],
      '#validate' => ['::validateForm'],
      '#submit' => ['::purgeUsersNowSubmit'],
      '#weight' => 100,
    ];

    return $actions;
  }

  /**
   * Submit handler for mass-account cancellation form.
   *
   * @see purge_users_config_form()
   */
  public function purgeUsersNowSubmit($form, FormStateInterface $form_state) {
    // Check if there is any user to purge.
    $ids_to_purge = $this->policyService->getPolicyUserIdsToPurge($this->entity);
    $ids_to_notify = $this->policyService->getPolicyUserIdsToNotify($this->entity);
    if (!$ids_to_purge && !$ids_to_notify) {
      $this->messenger()->addMessage($this->t(
        'No user account found in the system to purge or notify.'));
      return;
    }

    $form_state->setRedirect('purge_users.policy_confirm', ['policy_id' => $this->entity->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $status = parent::save($form, $form_state);

    if ($status) {
      $this->messenger()->addMessage($this->t('The purge users policy %label has been updated.', [
        '%label' => $this->entity->label(),
      ]));
    }

    // Clear the purge and notify queues.
    $this->policyService->clearQueues();

    return $status;
  }

  /**
   * Check to validate that the Purge Users Policy name does not already exist.
   *
   * @param string $name
   *   The machine name of the context to validate.
   *
   * @return bool
   *   TRUE on context name already exist, FALSE on context name not exist.
   */
  public function purgeUsersPolicyExists($name) {
    $entity = $this->entityTypeManager->getStorage('purge_users_policy')->loadByProperties(['name' => $name]);

    return (bool) $entity;
  }

  /**
   * Helper function to get the condition instances for the policy.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy $policy
   *   The policy entity.
   *
   * @return array
   *   An array of condition instances.
   */
  protected function getPolicyConditionsInstances(PurgeUsersPolicy $policy) {
    $instances = [];
    foreach ($policy->getConditions() as $id => $condition) {
      /** @var \Drupal\purge_users\PurgeUsersConditionInterface $instance */
      $instances[$id] = $this->manager->createInstance($condition['id'], $condition);
    }
    return $instances;
  }

  /**
   * Helper function to render the condition rows for the policy.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy $policy
   *   The policy entity.
   * @param array $instances
   *   An array of condition instances.
   *
   * @return array
   *   Condition rows rendered for the policy.
   */
  public function renderRows(PurgeUsersPolicy $policy, array $instances) {
    $configured_conditions = [];
    foreach ($policy->getConditions() as $id => $condition) {
      /** @var \Drupal\purge_users\PurgeUsersConditionInterface $instance */
      $instance = $instances[$id];

      $operations = $this->getOperations('entity.purge_users_policy.condition', [
        'policy_id' => $this->entity->id(),
        'plugin_id' => $condition['id'],
        'id' => $id,
      ]);

      $build = [
        '#type' => 'operations',
        '#links' => $operations,
      ];

      $configured_conditions[] = [
        'condition' => $instance->getPluginDefinition()['label'],
        'summary' => $instance->summary(),
        'operations' => [
          'data' => $build,
        ],
      ];
    }
    // Sort the policy conditions by label.
    uasort($configured_conditions, function ($a, $b) {
      // Spaceship operator: returns -1, 0, or 1.
      return $a['condition'] <=> $b['condition'];
    });
    return $configured_conditions;
  }

  /**
   * Helper function to load edit operations for a condition.
   *
   * @param string $route_name_base
   *   String representing the base of the route name for the conditions.
   * @param array $route_parameters
   *   Passing route parameter context to the helper function.
   *
   * @return array
   *   Set of operations associated with a condition.
   */
  protected function getOperations($route_name_base, array $route_parameters = []) {
    $edit_url = new Url($route_name_base . '.edit', $route_parameters);
    $delete_url = new Url($route_name_base . '.delete', $route_parameters);
    $operations = [];

    $operations['edit'] = [
      'title' => $this->t('Edit'),
      'url' => $edit_url,
      'weight' => 10,
      'attributes' => [
        'class' => ['use-ajax'],
        'data-dialog-type' => 'modal',
        'data-dialog-options' => Json::encode([
          'width' => 700,
        ]),
      ],
    ];
    $operations['delete'] = [
      'title' => $this->t('Delete'),
      'url' => $delete_url,
      'weight' => 100,
      'attributes' => [
        'class' => ['use-ajax'],
        'data-dialog-type' => 'modal',
        'data-dialog-options' => Json::encode([
          'width' => 700,
        ]),
      ],
    ];
    return $operations;
  }

  /**
   * Ajax callback that manages adding a condition.
   *
   * @param array $form
   *   Form definition of parent form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   State of the form.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   Returns the valid Ajax response from a modal window.
   */
  public function add(array &$form, FormStateInterface $form_state) {
    $condition_plugin_id = $form_state->getValue('condition');
    $definition = $this->manager->getDefinition($condition_plugin_id);
    $form_title = $definition['label'];
    $form_content = $this->formBuilder->getForm(
      ConditionEdit::class, $form_state->getValue('id'), $condition_plugin_id
    );
    $form_content['#attached']['library'][] = 'core/drupal.dialog.ajax';
    $url = Url::fromRoute('entity.purge_users_policy.condition.add', [
      'policy_id' => $this->entity->id(),
      'plugin_id' => $condition_plugin_id,
    ], ['query' => [FormBuilderInterface::AJAX_FORM_REQUEST => TRUE]]);
    $form_content['submit']['#attached']['drupalSettings']['ajax'][$form_content['submit']['#id']]['url'] = $url->toString();
    $response = new AjaxResponse();
    $response->addCommand(new OpenModalDialogCommand(
      $form_title, $form_content, ['width' => '700']
    ));
    return $response;
  }

  /**
   * Check if at least one condition is status related.
   *
   * @param array $condition_instances
   *   An array of the policy condition instances.
   */
  protected function hasStatusRelatedCondition(array $condition_instances): bool {
    $has_status_related_condition = FALSE;
    /** @var \Drupal\purge_users\PurgeUsersConditionInterface $instance */
    foreach ($condition_instances as $instance) {
      if ($instance->isStatusRelated()) {
        $has_status_related_condition = TRUE;
        break;
      }
    }
    return $has_status_related_condition;
  }

}
