<?php

namespace Drupal\purge_users\Form;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The general settings of the policy.
 */
abstract class PurgeUsersPolicyForm extends EntityForm {

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * Plugin manager for conditions.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $manager;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructor.
   */
  public function __construct(PluginManagerInterface $manager, LanguageManagerInterface $language_manager, FormBuilderInterface $formBuilder) {
    $this->manager = $manager;
    $this->languageManager = $language_manager;
    $this->formBuilder = $formBuilder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.condition'),
      $container->get('language_manager'),
      $container->get('form_builder')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $form['general'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('PURGE USERS POLICY'),
    ];

    $form['general']['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Policy Name'),
      '#default_value' => $this->entity->label(),
      '#required' => TRUE,
      '#description' => $this->t('Enter label for this context.'),
    ];

    $form['general']['id'] = [
      '#type' => 'machine_name',
      '#title' => $this->t('Machine name'),
      '#default_value' => $this->entity->id(),
      '#machine_name' => [
        'source' => ['general', 'label'],
        'exists' => [$this, 'purgeUsersPolicyExists'],
      ],
    ];

    return $form;
  }

  /**
   * Check to validate that the Purge Users Policy name does not already exist.
   *
   * @param string $name
   *   The machine name of the context to validate.
   *
   * @return bool
   *   TRUE on context name already exist, FALSE on context name not exist.
   */
  public function purgeUsersPolicyExists($name) {
    $entity = $this->entityTypeManager->getStorage('purge_users_policy')->loadByProperties(['name' => $name]);
    return (bool) $entity;
  }

}
