<?php

namespace Drupal\purge_users\Plugin\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\purge_users\PurgeUsersConditionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Do not purge users who are authors or commenters of existing content.
 *
 * @Condition(
 *   id = "purge_users:author_commenter",
 *   label = @Translation("Users authors or commenters"),
 *   description = @Translation("Do not purge users who are authors or commenters of existing content."),
 *   context_definitions = {
 *     "context" = @ContextDefinition("map", required = TRUE)
 *   }
 * )
 */
class AuthorCommenterCondition extends PurgeUsersConditionBase implements ContainerFactoryPluginInterface {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, $module_handler) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration, $plugin_id, $plugin_definition, $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    $context = $this->getContextValue('context');
    if (!empty($context)) {
      $query = $context['query'];
      $config = $this->getConfiguration();
      if ($config['enabled_do_not_purge_authors']) {
        $query->leftjoin('node_field_data', 'nfd', 'nfd.uid = u.uid');
        $query->isNull('nfd.uid');
      }
      if ($config['enabled_do_not_purge_commenters']) {
        // Check comment module activated to avoid table not found error.
        if ($this->moduleHandler->moduleExists('comment')) {
          $query->leftjoin('comment_field_data', 'cfd', 'cfd.uid = u.uid');
          $query->isNull('cfd.uid');
        }
      }
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'enabled_do_not_purge_authors' => FALSE,
      'enabled_do_not_purge_commenters' => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get configuration.
    $config = $this->getConfiguration();

    $form['enabled_do_not_purge_authors'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Do not purge users who are authors of existing content'),
      '#description' => $this->t('WARNING: Only applies to standard node entities. Custom entities are not covered.'),
      '#default_value' => $config['enabled_do_not_purge_authors'],
    ];

    $form['enabled_do_not_purge_commenters'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Do not purge users with existing comments'),
      '#default_value' => $config['enabled_do_not_purge_commenters'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['enabled_do_not_purge_authors'] = $form_state->getValue('enabled_do_not_purge_authors');
    $this->configuration['enabled_do_not_purge_commenters'] = $form_state->getValue('enabled_do_not_purge_commenters');
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    $authors = $this->configuration['enabled_do_not_purge_authors'];
    $commenters = $this->configuration['enabled_do_not_purge_commenters'];
    if ($authors && $commenters) {
      return $this->t('Do not purge users who are authors or commenters of existing content.');
    }
    elseif ($authors) {
      return $this->t('Do not purge users who are authors of existing content.');
    }
    elseif ($commenters) {
      return $this->t('Do not purge users with existing comments.');
    }
    else {
      return $this->t('⚠ Purge users even if they are authors or commenters of existing content.');
    }
  }

}
