<?php

namespace Drupal\purge_users\Plugin\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\purge_users\PurgeUsersConditionBase;

/**
 * Purge users who have been blocked for some amount of time.
 *
 * @Condition(
 *   id = "purge_users:blocked",
 *   label = @Translation("Users blocked for some time"),
 *   description = @Translation("Purge users who have been blocked for some amount of time."),
 *   status_related = TRUE,
 *   context_definitions = {
 *     "context" = @ContextDefinition("map", required = TRUE)
 *   }
 * )
 */
class BlockedUserCondition extends PurgeUsersConditionBase {

  use PolicyConditionTrait;

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    $context = $this->getContextValue('context');
    if (!empty($context)) {
      $config = $this->getConfiguration();
      $date = strtotime('-' . $config['user_blocked_value'] . ' ' . $config['user_blocked_period']);
      $mode = $context['mode'];
      if ($mode === 'notify') {
        $policy = $context['policy'];
        $date = $this->addPolicyNotificationDelay($policy, $date);
      }
      $query = $context['query'];
      $query->condition('u.created', $date, '<');
      $query->condition('u.login', $date, '<');
      $query->condition('u.login', 0, '!=');
      $query->condition('u.status', 0);
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'user_blocked_value' => 0,
      'user_blocked_period' => 'days',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get configuration.
    $config = $this->getConfiguration();

    $form['user_blocked_value'] = [
      '#type' => 'number',
      '#title' => $this->t('Value'),
      '#default_value' => $config['user_blocked_value'],
      '#prefix' => '<div class="purge-interval-selector clearfix">',
      '#attributes' => ['class' => ['purge-value']],
    ];

    $form['user_blocked_period'] = [
      '#title' => $this->t('Period'),
      '#type' => 'select',
      '#options' => [
        'days' => $this->t('Days'),
        'month' => $this->t('Months'),
        'year' => $this->t('Year'),
      ],
      '#default_value' => $config['user_blocked_period'],
      '#attributes' => ['class' => ['purge-period']],
      '#suffix' => '</div>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $login_value = (int) $form_state->getValue(['user_blocked_value']);
    $login_period = $form_state->getValue(['user_blocked_period']);
    // Validate to set the purge period to more than 1 day.
    if ($login_period === 'days' && $login_value < 1) {
      $form_state->setErrorByName('Period limit', $this->t('Purge period should be more than 1 day.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['user_blocked_value'] = $form_state->getValue('user_blocked_value');
    $this->configuration['user_blocked_period'] = $form_state->getValue('user_blocked_period');
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    return $this->t(
      'Purge users who have been blocked for @value @period.',
      [
        '@value' => $this->configuration['user_blocked_value'],
        '@period' => $this->configuration['user_blocked_period'],
      ]
    );
  }

}
