<?php

namespace Drupal\purge_users\Plugin\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\purge_users\PurgeUsersConditionBase;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Purge users who are not members of specific roles.
 *
 * @Condition(
 *   id = "purge_users:excluded_roles",
 *   label = @Translation("Users not members of specific roles"),
 *   description = @Translation("Purge users who are not members of specific roles."),
 *   context_definitions = {
 *     "context" = @ContextDefinition("map", required = TRUE)
 *   }
 * )
 */
class ExcludedRolesCondition extends PurgeUsersConditionBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration, $plugin_id, $plugin_definition, $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    $context = $this->getContextValue('context');
    if (!empty($context)) {
      $config = $this->getConfiguration();
      $excluded_user_roles = $config['purge_excluded_users_roles'];
      if (!empty($excluded_user_roles)) {
        $db = $context['db'];
        $not_exist_query = $db->select('user__roles', 'ur');
        $not_exist_query->fields('ur', ['entity_id']);
        $not_exist_query->where('ur.entity_id = u.uid');
        $not_exist_query->condition('ur.roles_target_id', $excluded_user_roles, 'IN');
        $query = $context['query'];
        $query->notExists($not_exist_query);
      }
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'purge_excluded_users_roles' => [],
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get configuration.
    $config = $this->getConfiguration();

    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    unset($roles[RoleInterface::ANONYMOUS_ID]);
    $roles_array = array_map(function ($item) {
      return $item->label();
    }, $roles);
    $exclude_role_array = $roles_array;
    unset($exclude_role_array[RoleInterface::AUTHENTICATED_ID]);

    $form['purge_excluded_users_roles'] = [
      '#title' => $this->t('Select roles to exclude from the purge'),
      '#description' => $this->t('Any user with one of the selected role will be excluded.'),
      '#type' => 'checkboxes',
      '#required' => TRUE,
      '#options' => $exclude_role_array,
      '#default_value' => is_array($config['purge_excluded_users_roles']) ? $config['purge_excluded_users_roles'] : [],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['purge_excluded_users_roles'] = $form_state->getValue('purge_excluded_users_roles');
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    $config = $this->getConfiguration();
    $selected_roles = is_array($config['purge_excluded_users_roles']) ? $config['purge_excluded_users_roles'] : [];
    if (empty($selected_roles)) {
      $roles_string = $this->t('No role selected.');
    }
    else {
      $all_roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
      $role_labels = array_map(
        fn($role) => $role->label(),
        array_intersect_key($all_roles, array_flip($selected_roles))
      );
      $roles_string = implode(', ', $role_labels);
    }
    return $this->t(
      'Purge users who are not members of the following roles: @roles',
      [
        '@roles' => $roles_string,
      ]
    );
  }

}
