<?php

namespace Drupal\purge_users\Plugin\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\purge_users\PurgeUsersConditionBase;

/**
 * Purge users whose account has not been activated for some amount of time.
 *
 * @Condition(
 *   id = "purge_users:inactive",
 *   label = @Translation("Users not activated for some time"),
 *   description = @Translation("Purge users whose account has not been activated for some amount of time."),
 *   status_related = TRUE,
 *   context_definitions = {
 *     "context" = @ContextDefinition("map", required = TRUE)
 *   }
 * )
 */
class InactiveUserCondition extends PurgeUsersConditionBase {

  use PolicyConditionTrait;

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    $context = $this->getContextValue('context');
    if (!empty($context)) {
      $config = $this->getConfiguration();
      $date = strtotime('-' . $config['user_inactive_value'] . ' ' . $config['user_inactive_period']);
      $mode = $context['mode'];
      if ($mode === 'notify') {
        $policy = $context['policy'];
        $date = $this->addPolicyNotificationDelay($policy, $date);
      }
      $query = $context['query'];
      $query->condition('u.created', $date, '<');
      $query->condition('u.login', 0);
      $query->condition('u.status', 0);
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'user_inactive_value' => 0,
      'user_inactive_period' => 'days',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get configuration.
    $config = $this->getConfiguration();

    $form['user_inactive_value'] = [
      '#type' => 'number',
      '#title' => $this->t('Value'),
      '#default_value' => $config['user_inactive_value'],
      '#prefix' => '<div class="purge-interval-selector clearfix">',
      '#attributes' => ['class' => ['purge-value']],
    ];

    $form['user_inactive_period'] = [
      '#title' => $this->t('Period'),
      '#type' => 'select',
      '#options' => [
        'days' => $this->t('Days'),
        'month' => $this->t('Months'),
        'year' => $this->t('Year'),
      ],
      '#default_value' => $config['user_inactive_period'],
      '#attributes' => ['class' => ['purge-period']],
      '#suffix' => '</div>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $login_value = (int) $form_state->getValue(['user_inactive_value']);
    $login_period = $form_state->getValue(['user_inactive_period']);
    // Validate to set the purge period to more than 1 day.
    if ($login_period === 'days' && $login_value < 1) {
      $form_state->setErrorByName('Period limit', $this->t('Purge period should be more than 1 day.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['user_inactive_value'] = $form_state->getValue('user_inactive_value');
    $this->configuration['user_inactive_period'] = $form_state->getValue('user_inactive_period');
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    return $this->t(
      'Purge users whose account has not been activated for @value @period.',
      [
        '@value' => $this->configuration['user_inactive_value'],
        '@period' => $this->configuration['user_inactive_period'],
      ]
    );
  }

}
