<?php

namespace Drupal\purge_users\Plugin\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\purge_users\PurgeUsersConditionBase;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Purge users who are members of specific roles only.
 *
 * @Condition(
 *   id = "purge_users:included_roles",
 *   label = @Translation("Users members of specific roles only"),
 *   description = @Translation("Purge users who are members of specific roles only."),
 *   context_definitions = {
 *     "context" = @ContextDefinition("map", required = TRUE)
 *   }
 * )
 */
class IncludedRolesCondition extends PurgeUsersConditionBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration, $plugin_id, $plugin_definition, $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    $context = $this->getContextValue('context');
    if (!empty($context)) {
      $config = $this->getConfiguration();
      $included_users_roles = $config['purge_included_users_roles'];
      if (!empty($included_users_roles)) {
        $query = $context['query'];
        $query->leftjoin('user__roles', 'ur', 'ur.entity_id = u.uid');
        $db = $context['db'];
        $exist_query = $db->select('user__roles', 'ur');
        $exist_query->fields('ur', ['entity_id']);
        $exist_query->where('ur.entity_id = u.uid');
        $exist_query->condition('ur.roles_target_id', $included_users_roles, 'IN');
        if (in_array(AccountInterface::AUTHENTICATED_ROLE, $included_users_roles)) {
          $db_or = $db->condition('OR');
          $db_or->isNull('ur.roles_target_id');
          $db_or->exists($exist_query);
          $query->condition($db_or);
        }
        else {
          $query->exists($exist_query);
        }
      }

    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'purge_included_users_roles' => [],
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get configuration.
    $config = $this->getConfiguration();

    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    unset($roles[RoleInterface::ANONYMOUS_ID]);
    $roles_array = array_map(function ($item) {
      return $item->label();
    }, $roles);

    $form['purge_included_users_roles'] = [
      '#title' => $this->t('Select roles to include in the purge'),
      '#description' => $this->t('Any user with one of the selected roles will be included, unless that user has a role among the one selected in the exclude roles field.'),
      '#type' => 'checkboxes',
      '#required' => TRUE,
      '#options' => $roles_array,
      '#default_value' => is_array($config['purge_included_users_roles']) ? $config['purge_included_users_roles'] : [],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['purge_included_users_roles'] = $form_state->getValue('purge_included_users_roles');
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    $config = $this->getConfiguration();
    $selected_roles = is_array($config['purge_included_users_roles']) ? $config['purge_included_users_roles'] : [];
    if (empty($selected_roles)) {
      $roles_string = $this->t('No role selected.');
    }
    else {
      $all_roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
      $role_labels = array_map(
        fn($role) => $role->label(),
        array_intersect_key($all_roles, array_flip($selected_roles))
      );
      $roles_string = implode(', ', $role_labels);
    }
    return $this->t(
      'Purge users who are members of the following roles only: @roles',
      [
        '@roles' => $roles_string,
      ]
    );
  }

}
