<?php

namespace Drupal\purge_users\Plugin\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\purge_users\PurgeUsersConditionBase;

/**
 * Purge users that have never logged in for some amount of time.
 *
 * @Condition(
 *   id = "purge_users:never_logged_in",
 *   label = @Translation("Users never logged in for some time"),
 *   description = @Translation("Purge users that have never logged in for some amount of time."),
 *   status_related = TRUE,
 *   context_definitions = {
 *     "context" = @ContextDefinition("map", required = TRUE)
 *   }
 * )
 */
class NeverLoggedInCondition extends PurgeUsersConditionBase {

  use PolicyConditionTrait;

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    $context = $this->getContextValue('context');
    if (!empty($context)) {
      $config = $this->getConfiguration();
      $never_login_date = strtotime('-' . $config['user_never_lastlogin_value'] . ' ' . $config['user_never_lastlogin_period']);
      $mode = $context['mode'];
      if ($mode === 'notify') {
        $policy = $context['policy'];
        $never_login_date = $this->addPolicyNotificationDelay($policy, $never_login_date);
      }
      $query = $context['query'];
      $query->condition('u.created', $never_login_date, '<');
      $query->condition('u.login', 0);
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'user_never_lastlogin_value' => 0,
      'user_never_lastlogin_period' => 'days',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get configuration.
    $config = $this->getConfiguration();

    $form['user_never_lastlogin_value'] = [
      '#type' => 'number',
      '#title' => $this->t('Value'),
      '#default_value' => $config['user_never_lastlogin_value'],
      '#prefix' => '<div class="purge-interval-selector clearfix">',
      '#attributes' => ['class' => ['purge-value']],
    ];

    $form['user_never_lastlogin_period'] = [
      '#title' => $this->t('Period'),
      '#type' => 'select',
      '#options' => [
        'days' => $this->t('Days'),
        'month' => $this->t('Months'),
        'year' => $this->t('Year'),
      ],
      '#default_value' => $config['user_never_lastlogin_period'],
      '#attributes' => ['class' => ['purge-period']],
      '#suffix' => '</div>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $login_never_value = (int) $form_state->getValue(['user_never_lastlogin_value']);
    $login_never_period = $form_state->getValue(['user_never_lastlogin_period']);
    // Validate to set the purge period to more than 1 day.
    if ($login_never_period === 'days' && $login_never_value < 1) {
      $form_state->setErrorByName('Period limit', $this->t('Purge period should be more than 1 day.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['user_never_lastlogin_value'] = $form_state->getValue('user_never_lastlogin_value');
    $this->configuration['user_never_lastlogin_period'] = $form_state->getValue('user_never_lastlogin_period');
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    return $this->t(
      'Purge users that have never logged in since their account creation for @value @period.',
      [
        '@value' => $this->configuration['user_never_lastlogin_value'],
        '@period' => $this->configuration['user_never_lastlogin_period'],
      ]
    );
  }

}
